# $Id: conn.bro,v 1.16 2005/06/19 06:35:41 vern Exp $

@load notice
@load hot
@load port-name
@load scan
@load netstats
@load conn-id

redef enum Notice += {
	SensitiveConnection,	# connection marked "hot"
	TerminatingConnection,	# connection will be terminated
	TerminatingConnectionIgnored,	# connection terminated disabled
};

const conn_closed = { TCP_CLOSED, TCP_RESET };

global have_FTP = F;	# if true, we've loaded ftp.bro
global have_SMTP = F;	# if true, we've loaded smtp.bro
global is_ftp_data_conn: function(c: connection): bool;

# Whether we're allowed (and/or are capable) to terminate connections
# using "rst".
global activate_terminate_connection = F &redef;

# Maps a given port on a given server's address to an RPC service.
# If we haven't loaded portmapper.bro, then it will be empty
# (and, ideally, queries to it would be optimized away ...).
global RPC_server_map: table[addr, port] of string;

global conn_file = open_log_file("conn") &redef;

function conn_state(c: connection, trans: transport_proto): string
	{
	local os = c$orig$state;
	local rs = c$resp$state;

	local o_inactive = os == TCP_INACTIVE || os == TCP_PARTIAL;
	local r_inactive = rs == TCP_INACTIVE || rs == TCP_PARTIAL;

	if ( trans == tcp )
		{
		if ( rs == TCP_RESET )
			{
			if ( os == TCP_SYN_SENT || os == TCP_SYN_ACK_SENT ||
			     (os == TCP_RESET &&
			      c$orig$size == 0 && c$resp$size == 0) )
				return "REJ";
			else if ( o_inactive )
				return "RSTRH";
			else
				return "RSTR";
			}
		else if ( os == TCP_RESET )
			return r_inactive ? "RSTOS0" : "RSTO";
		else if ( rs == TCP_CLOSED && os == TCP_CLOSED )
			return "SF";
		else if ( os == TCP_CLOSED )
			return r_inactive ? "SH" : "S2";
		else if ( rs == TCP_CLOSED )
			return o_inactive ? "SHR" : "S3";
		else if ( os == TCP_SYN_SENT && rs == TCP_INACTIVE )
			return "S0";
		else if ( os == TCP_ESTABLISHED && rs == TCP_ESTABLISHED )
			return "S1";
		else
			return "OTH";
		}

	else if ( trans == udp )
		{
		if ( os == UDP_ACTIVE )
			return rs == UDP_ACTIVE ? "SF" : "S0";
		else
			return rs == UDP_ACTIVE ? "SHR" : "OTH";
		}

	else
		return "OTH";
	}

function conn_size(e: endpoint, trans: transport_proto): string
	{
	if ( e$size > 0 || (trans == tcp && e$state == TCP_CLOSED) )
		return fmt("%d", e$size);
	else
		### should return 0 for TCP_RESET that went through TCP_CLOSED
		return "?";
	}

function service_name(c: connection): string
	{
	local p = c$id$resp_p;
	local trans = get_port_transport_proto(p);

	if ( p in port_names )
		return port_names[p];
	else
		return "other";
	}

const state_graphic = {
	["OTH"] = "?>?", ["REJ"] = "[",
	["RSTO"] = ">]", ["RSTOS0"] = "}]", ["RSTR"] = ">[", ["RSTRH"] = "<[",
	["S0"] = "}", ["S1"] = ">", ["S2"] = "}2", ["S3"] = "}3",
	["SF"] = ">", ["SH"] = ">h", ["SHR"] = "<h",
};

function full_id_string(c: connection): string
	{
	local id = c$id;
	local trans = get_port_transport_proto(id$orig_p);
	local state = conn_state(c, trans);
	local state_gr = state_graphic[state];
	local service = c$service == "" ? service_name(c) : c$service;

	if ( state == "S0" || state == "S1" || state == "REJ" )
		return fmt("%s %s %s/%s %s", id$orig_h, state_gr,
				id$resp_h, service, c$addl);

	else
		return fmt("%s %sb %s %s/%s %sb %.1fs %s",
			id$orig_h, conn_size(c$orig, trans),
			state_gr, id$resp_h, service,
			conn_size(c$resp, trans), c$duration, c$addl);
	}

# The sets are indexed by the complete hot messages.
global hot_conns_reported: table[conn_id] of set[string];

function log_hot_conn(c: connection)
	{
	if ( c$id !in hot_conns_reported )
		{
		local empty_set: set[string];
		hot_conns_reported[c$id] = empty_set;
		}

	local msg = full_id_string(c);

	if ( msg !in hot_conns_reported[c$id] )
		{
		NOTICE([$note=SensitiveConnection, $conn=c,
			$msg=fmt("hot: %s", full_id_string(c))]);

		add hot_conns_reported[c$id][msg];
		}
	}

function terminate_connection(c: connection)
	{
	if ( activate_terminate_connection )
		{
		local id = c$id;
		local local_init = is_local_addr(id$orig_h);

		local term_cmd = fmt("./rst %s -n 32 -d 20 %s %d %d %s %d %d",
					local_init ? "-R" : "",
					id$orig_h, id$orig_p, get_orig_seq(id),
					id$resp_h, id$resp_p, get_resp_seq(id));

		if ( reading_live_traffic() )
			system(term_cmd);
		else
			NOTICE([$note=TerminatingConnection, $conn=c,
				$msg=term_cmd, $sub="first termination command"]);

		term_cmd = fmt("./rst %s -r 2 -n 4 -s 512 -d 20 %s %d %d %s %d %d",
				local_init ? "-R" : "",
				id$orig_h, id$orig_p, get_orig_seq(id),
				id$resp_h, id$resp_p, get_resp_seq(id));

		if ( reading_live_traffic() )
			system(term_cmd);
		else
			NOTICE([$note=TerminatingConnection, $conn=c,
				$msg=term_cmd, $sub="second termination command"]);

		NOTICE([$note=TerminatingConnection, $conn=c,
			$msg=fmt("terminating %s", full_id_string(c))]);
		}

	else
		NOTICE([$note=TerminatingConnectionIgnored, $conn=c,
			$msg=fmt("ignoring request to terminate %s",
					full_id_string(c))]);
	}

function record_connection(f: file, c: connection)
	{
	local id = c$id;
	local local_init = is_local_addr(id$orig_h);

	local local_addr = local_init ? id$orig_h : id$resp_h;
	local remote_addr = local_init ? id$resp_h : id$orig_h;

	local flags = local_init ? "L" : "X";

	local trans = get_port_transport_proto(id$orig_p);
	local duration: string;
	if ( trans == tcp )
		{
		if ( c$orig$state in conn_closed || c$resp$state in conn_closed )
			duration = fmt("%.06f", c$duration);
		else
			duration = "?";
		}
	else
		duration = fmt("%.06f", c$duration);

	local addl = c$addl == "" ? "" : cat(" ", c$addl);
	local s = c$service == "" ? service_name(c) : c$service;
	local state = conn_state(c, trans);

@ifdef ( estimate_flow_size_and_remove )
	# Annotate connection with separately-estimated size, if present.
	local orig_est = estimate_flow_size_and_remove(id, T);
	local resp_est = estimate_flow_size_and_remove(id, F);

	if ( orig_est$have_est )
		addl = fmt("%s olower=%.0fMB oupper=%.0fMB oincon=%s", addl,
				orig_est$lower / 1e6, orig_est$upper / 1e6,
				orig_est$num_inconsistent);

	if ( resp_est$have_est )
		addl = fmt("%s rlower=%.0fMB rupper=%.0fMB rincon=%s", addl,
				resp_est$lower / 1e6, resp_est$upper / 1e6,
				resp_est$num_inconsistent);
@endif

	print f, fmt("%.6f %s %s %s %s %d %d %s %s %s %s %s%s",
		c$start_time, duration, id$orig_h, id$resp_h, s,
		id$orig_p, id$resp_p, trans,
		conn_size(c$orig, trans), conn_size(c$resp, trans),
		state, flags, addl);

	if ( c$hot > 0 )
		log_hot_conn(c);
	}

function determine_service(c: connection)
	{
	if ( have_FTP && is_ftp_data_conn(c) )
		c$service = port_names[20/tcp];

	else if ( [c$id$resp_h, c$id$resp_p] in RPC_server_map )
		# Alternatively, perhaps this should be stored in $addl
		# rather than $service, so the port number remains
		# visible .... ?
		c$service = RPC_server_map[c$id$resp_h, c$id$resp_p];

	else if ( c$orig$state == TCP_INACTIVE )
		{
		# We're seeing a half-established connection.  Use the
		# service of the originator if it's well-known and the
		# responder isn't.
		if ( c$id$resp_p !in port_names && c$id$orig_p in port_names )
			c$service = port_names[c$id$orig_p];
		}
	}

event connection_established(c: connection)
	{
	determine_service(c);

	local is_reverse_scan = (c$orig$state == TCP_INACTIVE);
	check_scan(c, T, is_reverse_scan);

	local trans = get_port_transport_proto(c$id$orig_p);
	if ( trans == tcp && ! is_reverse_scan && use_TRW_algorithm )
		check_TRW_scan(c, conn_state(c, trans), F);

	check_hot(c, CONN_ESTABLISHED);

	if ( c$hot > 0 )
		log_hot_conn(c);
	}

event partial_connection(c: connection)
	{
	check_scan(c, T, F);

	if ( c$orig$state == TCP_PARTIAL && c$resp$state == TCP_INACTIVE )
		# This appears to be a stealth scan.  Don't do hot-checking
		# as there wasn't an established connection.
		;
	else
		{
		check_hot(c, CONN_ESTABLISHED);
		check_hot(c, APPL_ESTABLISHED);	# assume it's been established
		}

	if ( c$hot > 0 )
		log_hot_conn(c);
	}

event connection_attempt(c: connection)
	{
	determine_service(c);

	check_scan(c, F, F);

	local trans = get_port_transport_proto(c$id$orig_p);
	if ( trans == tcp && use_TRW_algorithm )
		check_TRW_scan(c, conn_state(c, trans), F);

	check_spoof(c);
	check_hot(c, CONN_ATTEMPTED);
	}

event connection_finished(c: connection)
	{
	if ( c$orig$size == 0 || c$resp$size == 0 )
		# Hard to get excited about this - not worth logging again.
		c$hot = 0;
	else
		check_hot(c, CONN_FINISHED);
	}

event connection_half_finished(c: connection)
	{
	# Half connections never were "established", so do scan-checking here.
	# (But don't do hot-checking, since the connection was never
	# established; it's likely the result of stealth-scanning.)
	determine_service(c);
	check_scan(c, F, F);
	check_hot(c, CONN_ATTEMPTED);
	}

event connection_rejected(c: connection)
	{
	determine_service(c);

	local is_reverse_scan =
		c$orig$state == TCP_RESET || c$orig$state == TCP_SYN_ACK_SENT;

	check_scan(c, F, is_reverse_scan);

	local trans = get_port_transport_proto(c$id$orig_p);
	if ( trans == tcp && use_TRW_algorithm )
		check_TRW_scan(c, conn_state(c, trans), is_reverse_scan);

	check_hot(c, CONN_REJECTED);
	}

event connection_reset(c: connection)
	{
	if ( c$orig$state == TCP_INACTIVE || c$resp$state == TCP_INACTIVE )
		# We never heard from one side - that looks like a scan.
		check_scan(c, c$orig$size + c$resp$size > 0,
				c$orig$state == TCP_INACTIVE);

	check_hot(c, CONN_FINISHED);
	}

event connection_pending(c: connection)
	{
	if ( c$orig$state in conn_closed &&
	     (c$resp$state == TCP_INACTIVE || c$resp$state == TCP_PARTIAL) )
		# This is a stray FIN or RST - don't bother reporting.
		return;

	if ( c$orig$state == TCP_RESET || c$resp$state == TCP_RESET )
		# We already reported this connection when the RST
		# occurred.
		return;

	check_hot(c, CONN_FINISHED);
	}

function connection_gone(c: connection, gone_type: string)
	{
	if ( c$orig$size == 0 || c$resp$size == 0 )
		{
		if ( c$orig$state == TCP_RESET && c$resp$state == TCP_INACTIVE)
			# A bare RST, no other context.  Ignore it.
			return;

		# Hard to get excited about this - not worth logging again,
		# per connection_finished().
		c$hot = 0;
		}
	else
		check_hot(c, CONN_TIMEOUT);
	}

event connection_state_remove(c: connection)
	{
	local os = c$orig$state;
	local rs = c$resp$state;

	if ( os == TCP_ESTABLISHED && rs == TCP_ESTABLISHED )
		# It was still active, no summary generated.
		connection_gone(c, "remove");

	else if ( (os == TCP_CLOSED || rs == TCP_CLOSED) &&
		  (os == TCP_ESTABLISHED || rs == TCP_ESTABLISHED) )
		# One side has closed, the other hasn't - it's in state S2
		# or S3, hasn't been reported yet.
		connection_gone(c, "remove");

	record_connection(conn_file, c);

	delete hot_conns_reported[c$id];
	}
