# $Id: firewall.bro,v 1.1 2005/03/17 09:18:46 vern Exp $
#
# Firewall-like rules.

@load notice
@load conn

module Firewall;

export {
	type action: enum { ALLOW, DENY };
	type cmp: enum { EQ, NE };

	type rule: record {
		label: string &default = "<no-label>";
		orig_net: subnet &default = 0.0.0.0/0;
		orig_net_cmp: cmp &default = EQ;
		orig_p: port &default = 0/tcp;
		orig_p_cmp: cmp &default = EQ;
		resp_net: subnet &default = 0.0.0.0/0;
		resp_net_cmp: cmp &default = EQ;
		resp_p: port &default = 0/tcp;
		resp_p_cmp: cmp &default = EQ;
		prot: transport_proto &default = unknown_transport;
		prot_cmp: cmp &default = EQ;
		state: string &default = "";
		state_cmp: cmp &default = EQ;

		action: action &default = ALLOW;
	};

	redef enum Notice += {
		DenyRuleMatched
	};

	global begin: function();
	global match_rule: function(c: connection, r: rule);
}

global log_file = open_log_file("firewall");

global stop_matching = F;

function do_match(c: connection, r: rule): bool
	{
	if ( r$orig_net_cmp == EQ )
		{
		if ( c$id$orig_h !in r$orig_net )
			return F;
		}
	else
		if ( c$id$orig_h in r$orig_net )
			return F;

	if ( r$resp_net_cmp == EQ )
		{
		if ( c$id$resp_h !in r$resp_net )
			return F;
		}
	else
		if ( c$id$resp_h in r$resp_net )
			return F;

	if ( r$orig_p != 0/tcp )
		{
		if ( r$orig_p_cmp == EQ )
			{
			if ( c$id$orig_p != r$orig_p )
				return F;
			}
		else
			if ( c$id$orig_p == r$orig_p )
				return F;
		}

	if ( r$resp_p != 0/tcp )
		{
		if ( r$resp_p_cmp == EQ )
			{
			if ( c$id$resp_p != r$resp_p )
				return F;
			}
		else
			if ( c$id$resp_p == r$resp_p )
				return F;
		}

	if ( r$state != "" )
		{
		local state = conn_state(c, get_port_transport_proto(c$id$orig_p));
		if ( r$state_cmp == EQ )
			{
			if ( state != r$state )
				return F;
			}
		else
			if ( state == r$state )
				return F;
		}

	if ( r$prot != unknown_transport )
		{
		local proto = get_port_transport_proto(c$id$orig_p);
		if ( r$prot_cmp == EQ )
			{
			if ( proto != r$prot )
				return F;
			}
		else
			if ( proto == r$prot )
				return F;
		}

	return T;
	}

function begin()
	{
	stop_matching = F;
	}

event bro_init()
	{
	set_buf(log_file, F);
	}

function match_rule(c: connection, r: rule)
	{
	if ( stop_matching )
		return;

	if ( do_match(c, r) )
		{
		stop_matching = T;

		if ( r$action == DENY )
			{
			NOTICE([$note=DenyRuleMatched,
					$msg=fmt("%s: fw deny rule %s matched",
						id_string(c$id), r$label),
						$conn=c, $sub=r$label]);
			append_addl(c, fmt("<%s>", r$label));
			record_connection(log_file, c);
			}
		}
	}
