# $Id: notice.bro,v 1.15 2005/03/21 07:38:53 vern Exp $

global use_tagging = F &redef;

type Notice: enum {
	Unspecified,	# place-holder - shouldn't be used
};

type notice_info: record {
	note: Notice;
	msg: string;
	sub: string &optional;	# sub-message

	conn: connection &optional;	# connection associated with notice
	iconn: icmp_conn &optional;	# associated ICMP "connection"
	id: conn_id &optional;	# connection-ID, if we don't have a connection handy
	src: addr &optional;	# source address, if we don't have a connection
	dst: addr &optional;	# destination address

	p: port &optional;	# associated port, if we don't have a conn.

	# The following are detailed attributes that are associated with some
	# notices, but not all.

	user: string &optional;

	filename: string &optional;

	method: string &optional;
	URL: string &optional;

	n: count &optional;	# associated count, or perhaps status code

	# Source that raised this notice (automatically set).
	src_peer: event_peer &optional;
};

type NoticeAction: enum {
	# Similar to WeirdAction in weird.bro.
	NOTICE_IGNORE, NOTICE_FILE,
	NOTICE_ALARM_ALWAYS, NOTICE_ALARM_PER_CONN,
	NOTICE_ALARM_PER_ORIG, NOTICE_ALARM_ONCE,
	NOTICE_EMAIL, NOTICE_PAGE,
};

type notice_policy_item: record {
	result: NoticeAction;
	pred: function(n: notice_info): bool;
	priority: count;
};

global notice_policy: set[notice_policy_item] = {
	[$pred(n: notice_info) = { return T; },
	 $result = NOTICE_ALARM_ALWAYS,
	 $priority = 0],
} &redef;


# Variables the control email notification.
global mail_script = "mail_notice.sh" &redef;
global mail_dest = "" &redef;
global mail_page_dest = "bro-page" &redef;


# Table that maps notices into a function that should be called
# to determine the action.
global notice_action_filters:
	table[Notice] of function(n: notice_info): NoticeAction &redef;


# Each notice has a unique ID associated with it.
global notice_id = 0;

function ignore_notice(n: notice_info): NoticeAction
	{
	return NOTICE_IGNORE;
	}

function file_notice(n: notice_info): NoticeAction
	{
	return NOTICE_FILE;
	}

function send_email_notice(n: notice_info): NoticeAction
	{
	return NOTICE_EMAIL;
	}

function send_page_notice(n: notice_info): NoticeAction
	{
	return NOTICE_PAGE;
	}

global notice_tallies: table[string] of count &default = 0;

function tally_notice(s: string)
	{
	++notice_tallies[s];
	}

function tally_notice_type(n: notice_info): NoticeAction
	{
	tally_notice(fmt("%s", n$note));
	return NOTICE_FILE;
	}

event bro_done()
	{
	for ( s in notice_tallies )
		{
		local n = notice_tallies[s];
		alarm fmt("%s (%d time%s)", s, n, n > 1 ? "s" : "");
		}
	}

global notice_file = open_log_file("notice") &redef;

# This handler is useful for processing notices after the notice filters
# have been applied and yielded an NoticeAction.
#
# It's tempting to make the default handler do the logging and
# printing to notice_file, rather than NOTICE.  I hesitate to do that,
# though, because it perhaps could slow down notification, because
# in the absence of event priorities, the event would have to wait
# behind any other already-queued events.
event notice_action(n: notice_info, action: NoticeAction)
	{
	}

# Given a string, returns an escaped version suitable for being
# printed in the colon-separated notice format.  This means that
# (1) any colons are escaped using '\', and (2) any '\'s are
# likewise escaped.
function notice_escape(s: string): string
	{
	if ( use_tagging )
		{
		s = subst_string(s, "\\", "\\\\");
		return subst_string(s, " ", "\\ ");
		}
	else
		{
		s = subst_string(s, "\\", "\\\\");
		return subst_string(s, ":", "\\:");
		}
	}

# Hack to suppress duplicate notice_actions for remote notices.
global suppress_notice_action = F;

function add_info(cur_info: string, tag: string, val: string): string
	{
	val = notice_escape(val);

	if ( use_tagging )
		{
		if ( cur_info == "" )
			return fmt("%s=%s", tag, val);
		else
			return fmt("%s %s=%s", cur_info, tag, val);
		}
	else
		{
		if ( cur_info == "" )
			return val;
		else
			return fmt("%s:%s", cur_info, val);
		}
	}

function add_nil_info(cur_info: string): string
	{
	if ( use_tagging )
		return cur_info;
	else
		{
		if ( cur_info == "" )
			return ":";
		else
			return fmt("%s:", cur_info);
		}
	}

function email_notice(n: notice_info, action: NoticeAction)
	{
	if ( ! reading_live_traffic() || mail_dest == "" )
		return;

	# Choose destination address based on action type.
	local destination = (action == NOTICE_EMAIL) ?
		mail_dest : mail_page_dest;

	# The contortions here ensure that the arguments to the mail
	# script will not be confused.  Re-evaluate if 'system' is reworked.
	local mail_cmd =
		fmt("echo \"%s\" | %s %s %s",
			n$msg, mail_script, n$note, destination);

	system(mail_cmd);
	}


function NOTICE(n: notice_info)
	{
	# Fill in some defaults.
	if ( ! n?$id && n?$conn )
		n$id = n$conn$id;

	if ( ! n?$src && n?$id )
		n$src = n$id$orig_h;
	if ( ! n?$dst && n?$id )
		n$dst = n$id$resp_h;

	if ( ! n?$p && n?$id )
		n$p = n$id$resp_p;

	if ( ! n?$src && n?$iconn )
		n$src = n$iconn$orig_h;
	if ( ! n?$dst && n?$iconn )
		n$dst = n$iconn$resp_h;

	if ( ! n?$src_peer )
		n$src_peer = get_event_peer();

	local action = match n using notice_policy;
	local n_id = "" &redef;

	if ( n$note in notice_action_filters )
		action = notice_action_filters[n$note](n);

	if ( action != NOTICE_IGNORE )
		{
		local info = "";

		info = add_info(info, "t", fmt("%.06f", network_time()));
		info = add_info(info, "no", fmt("%s", n$note));
		info = add_info(info, "na", fmt("%s", action));

		if ( is_remote_event() )
			{
			if ( n$src_peer$descr != "" )
				info = add_info(info, "es", n$src_peer$descr);
			else
				info = add_info(info, "es",
					fmt("%s/%s", n$src_peer$host,
						n$src_peer$p));
			}
		else
			info = add_nil_info(info);

		if ( n?$src )
			info = add_info(info, "sa", fmt("%s", n$src));
		else
			info = add_nil_info(info);
		if ( n?$id && n$id$orig_h == n$src )
			info = add_info(info, "sp", fmt("%s", n$id$orig_p));
		else
			info = add_nil_info(info);

		if ( n?$dst )
			info = add_info(info, "da", fmt("%s", n$dst));
		else
			info = add_nil_info(info);
		if ( n?$id && n$id$resp_h == n$dst )
			info = add_info(info, "dp", fmt("%s", n$id$resp_p));
		else
			info = add_nil_info(info);

		if ( n?$user )
			info = add_info(info, "user", n$user);
		else
			info = add_nil_info(info);
		if ( n?$filename )
			info = add_info(info, "file", n$filename);
		else
			info = add_nil_info(info);
		if ( n?$method )
			info = add_info(info, "method", n$method);
		else
			info = add_nil_info(info);
		if ( n?$URL )
			info = add_info(info, "url", n$URL);
		else
			info = add_nil_info(info);

		if ( n?$n )
			info = add_info(info, "num", fmt("%s", n$n));
		else
			info = add_nil_info(info);

		if ( n?$msg )
			info = add_info(info, "msg", n$msg);
		else
			info = add_nil_info(info);
		if ( n?$sub )
			info = add_info(info, "sub", n$sub);
		else
			info = add_nil_info(info);

		if ( n?$conn )
			{
			local tag_id = fmt("@%s", ++notice_id);
			n$conn$addl = fmt("%s %s", n$conn$addl, tag_id);
			info = add_info(info, "tag", tag_id);
			}

		print notice_file, info;

		if ( action != NOTICE_FILE )
			{
			if ( action == NOTICE_EMAIL || action == NOTICE_PAGE )
				email_notice(n, action);

			if ( use_tagging )
				alarm info;

			else
				{
				local descr = "";
				if ( is_remote_event() )
					{
					if ( n$src_peer$descr != "" )
						descr = fmt("<%s> ", n$src_peer$descr);
					else
						descr = fmt("<%s:%s> ", n$src_peer$host,
									n$src_peer$p);
					}

				alarm fmt("%s %s%s", n$note, descr, n$msg);
				}
			}
		}

@ifdef ( IDMEF_support )
	if ( n?$id )
		generate_idmef(n$id$orig_h, n$id$orig_p,
			       n$id$resp_h, n$id$resp_p);
@endif

	event notice_action(n, action);
	}
