# $Id: remote.bro,v 1.9 2005/03/17 09:20:14 vern Exp $
#
# Connect to remote Bros and request some of their events.

@load notice

module Remote;

export {
	const default_port_ssl = 47756/tcp &redef;
	const default_port_clear = 47757/tcp &redef;

	# A remote peer to which we would like to talk.
	# If there's no entry for a peer, it may still connect
	# and request state, but not send us any.
	type Destination : record {
		# Destination endpoint.
		host: addr;
		p: port &optional;

		# Events requested from remote side.
		events: pattern &optional;

		# Whether we are going to connect (rather than waiting
		# for the other sie to connect to us).
		connect: bool &default = F;

		# If disconnected, reconnect after this many seconds.
		retry: interval &default = 0 secs;

		# Whether to accept state.
		accept_state: bool &default = T;

		# Whether to synchronize IDs.
		sync: bool &default = T;

		# If not set, no capture filter is sent.
		# If set to "", the default cature filter is sent.
		capture_filter: string &optional;

		# Whether to use SSL-based communication.
		ssl: bool &default = F;

		# Take-over state from this host
		# (activated by loading hand-over.bro)
		hand_over: bool &default = F;

		# Set when connected.
		peer: event_peer &optional;
		connected: bool &default = F;
	};

	const destinations: table[string] of Destination &redef;

	# redef destinations += {
	#	["foo"] = [$host = foo.bar.com, $events = /.*/, $connect=T, $retry = 60 secs, $ssl=T]
	# };

	# If true, re-raise received notice's locally.
	global report_remote_notices = T &redef;

	# Write log message into remote.log
	global do_log: function(p: event_peer, msg: string);

	global pending_peers: table[peer_id] of Destination;
	global connected_peers: table[peer_id] of Destination;

}
global remote_log = open_log_file("remote");

function do_log(p: event_peer, msg: string)
	{
	print remote_log,
		fmt("%.6f %s/%d %s", current_time(), p$host, p$p, msg);
	}

event bro_init()
	{
	set_buf(remote_log, F);

	for ( tag in destinations )
		{
		local dst = destinations[tag];
		local p = dst$ssl ? default_port_ssl : default_port_clear;

		if ( ! dst$connect )
			next;

		if ( dst?$p )
			p = dst$p;

		local id = connect(dst$host, p, dst$retry, dst$ssl);

		if ( id == PEER_ID_NONE )
			print remote_log,
				fmt("%.6f %s/%d can't trigger connect",
					current_time(), dst$host, p);

		pending_peers[id] = dst;
		}
	}

function setup_peer(p: event_peer, dst: Destination)
	{
	if ( dst?$events )
		{
		do_log(p, fmt("requesting events matching %s", dst$events));
		request_remote_events(p, dst$events);
		}

	if ( dst?$capture_filter )
		{
		local filter = dst$capture_filter;
		if ( filter == "" )
			filter = default_pcap_filter;

		do_log(p, fmt("sending capture_filter: %s", filter));
		send_capture_filter(p, filter);
		}

	if ( dst$accept_state )
		{
		do_log(p, "accepting state");
		set_accept_state(p, T);
		}

	if ( dst$sync )
		{
		do_log(p, "requesting synchronized state");
		request_remote_sync(p);
		}

	dst$peer = p;
	dst$connected = T;
	connected_peers[p$id] = dst;
	}

event remote_connection_established(p: event_peer)
	{
	do_log(p, "connection established.");

	if ( p$id in pending_peers )
		{
		# We issued the connect.
		local dst = pending_peers[p$id];
		setup_peer(p, dst);
		delete pending_peers[p$id];
		}
	else
		{ # The other side connected to us.
		for ( i in destinations )
			{
			dst = destinations[i];
			if ( dst$host == p$host )
				setup_peer(p, dst);
			}
		}

	complete_handshake(p);
	}

event remote_connection_closed(p: event_peer)
	{
	do_log(p, "connection closed.");

	if ( p$id in connected_peers )
		{
		local dst = connected_peers[p$id];
		dst$connected = F;

		delete connected_peers[p$id];

		if ( dst$retry != 0secs )
			# The core will retry.
			pending_peers[p$id] = dst;
		}
	}

event remote_state_inconsistency(operation: string, id: string,
				expected_old: string, real_old: string)
	{
	print remote_log,
		fmt("%.6f state inconsistency: %s should be %s but is %s before %s",
			network_time(), id, expected_old, real_old, operation);
	}

event notice_action(n: notice_info, action: NoticeAction)
	{
	# Forward remote alarms to our local system.
	if ( report_remote_notices && is_remote_event() )
		{
		# Don't raise this event again.
		suppress_notice_action = T;
		NOTICE(n);
		suppress_notice_action = F;
		}
	}
