# $Id: ssl.bro,v 1.4 2004/11/02 01:25:50 tierney Exp $

@load notice
@load conn
@load weird
@load ssl-ciphers
@load ssl-errors

global ssl_log = open_log_file("ssl") &redef;

redef enum Notice += {
	SSL_X509Violation,	# blanket X509 error
	SSL_SessConIncon,	# session data not consistent with connection
};


const SSLv20 = 0x0002;
const SSLv30 = 0x0300;
const SSLv31 = 0x0301;

# Handshake types.
const SSL_CLIENT_HELLO = 0;
const SSL_SERVER_HELLO = 0;
const SSL_CLIENT_MASTER_KEY = 0;

# If true, bro stores the client and server cipher specs and performs
# additional tests.  This costs an extra amount of memory (normally
# only for a short time) but enables detecting of non-intersecting
# cipher sets, for example.
const ssl_compare_cipherspecs = T &redef;

# Whether to analyze certificates seen in SSL connections.
const ssl_analyze_certificates = T &redef;

# If we analyze SSL certificates, we can choose to store them.
const ssl_store_certificates = T &redef;

# Path where we dump the certificates into.  If it's empty,
# use the current directory.
const ssl_store_cert_path = "" &redef;

# If we analyze SSL certificates, we can choose to verify them.
const ssl_verify_certificates = T &redef;

# This is the path where OpenSSL looks after the trusted certificates.
# If empty, the default path will be used.
const x509_trusted_cert_path = "" &redef;

# The maxiumum size in bytes for an SSL cipherspec.  If we see a packet that
# has bigger cipherspecs, we warn and won't do a comparisons of cipherspecs.
const ssl_max_cipherspec_size = 45 &redef;

# Whether to store key-material exchanged in the handshaking phase.
const ssl_store_key_material = T &redef;

# NOTE: this is a 'local' port format for your site
# --- well-known ports for ssl ---------
redef capture_filters += {
	["ssl"] = "tcp port 443",
	["nntps"] = "tcp port 563",
	["imap4-ssl"] = "tcp port 585",
	["sshell"] = "tcp port 614",
	["ldaps"] = "tcp port 636",
	["ftps-data"] = "tcp port 989",
	["ftps"] = "tcp port 990",
	["telnets"] = "tcp port 992",
	["imaps"] = "tcp port 993",
	["ircs"] = "tcp port 994",
	["pop3s"] = "tcp port 995"
};

# --- Weak Cipher Demo -------------

const myWeakCiphers: set[count] = {
	SSLv20_CK_RC4_128_EXPORT40_WITH_MD5,
	SSLv20_CK_RC2_128_CBC_EXPORT40_WITH_MD5,
	SSLv20_CK_DES_64_CBC_WITH_MD5,

	SSLv3x_NULL_WITH_NULL_NULL,
	SSLv3x_RSA_WITH_NULL_MD5,
	SSLv3x_RSA_WITH_NULL_SHA,
	SSLv3x_RSA_EXPORT_WITH_RC4_40_MD5,
	SSLv3x_RSA_EXPORT_WITH_RC2_CBC_40_MD5,
	SSLv3x_RSA_EXPORT_WITH_DES40_CBC_SHA,
	SSLv3x_RSA_WITH_DES_CBC_SHA,

	SSLv3x_DH_DSS_EXPORT_WITH_DES40_CBC_SHA,
	SSLv3x_DH_DSS_WITH_DES_CBC_SHA,
	SSLv3x_DH_RSA_EXPORT_WITH_DES40_CBC_SHA,
	SSLv3x_DH_RSA_WITH_DES_CBC_SHA,
	SSLv3x_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA,
	SSLv3x_DHE_DSS_WITH_DES_CBC_SHA,
	SSLv3x_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA,
	SSLv3x_DHE_RSA_WITH_DES_CBC_SHA,

	SSLv3x_DH_anon_EXPORT_WITH_RC4_40_MD5,
	SSLv3x_DH_anon_WITH_RC4_128_MD5,
	SSLv3x_DH_anon_EXPORT_WITH_DES40_CBC_SHA,
	SSLv3x_DH_anon_WITH_DES_CBC_SHA,
	SSLv3x_DH_anon_WITH_3DES_EDE_CBC_SHA,
	SSLv3x_FORTEZZA_KEA_WITH_NULL_SHA
};

const x509_ignore_errors: set[int] = {
	X509_V_OK,
	X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE
};

const x509_hot_errors: set[int] = {
	X509_V_ERR_CRL_SIGNATURE_FAILURE,
	X509_V_ERR_CERT_NOT_YET_VALID,
	X509_V_ERR_CERT_HAS_EXPIRED,
	X509_V_ERR_CERT_REVOKED,
	X509_V_ERR_SUBJECT_ISSUER_MISMATCH,
	# X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE	# for testing
};

redef weird_action += {
	["SSLv2: Unknown CIPHER-SPEC in CLIENT-HELLO!"]	= WEIRD_IGNORE,
	["SSLv2: Client has CipherSpecs > MAX_CIPHERSPEC_SIZE"]	= WEIRD_IGNORE,
	["unexpected_SSLv3_record"]	= WEIRD_IGNORE,
	["SSLv3_data_without_full_handshake"]	= WEIRD_IGNORE,
};


global SSL_cipherCount: table[count] of count &default = 0;

type ssl_connection_info: record {
	id: count;			# the log identifier number
	connection_id: conn_id;		# IP connection information
	version: string;		# version assosciated with connection
	client_cert: X509;
	server_cert: X509;
	id_index: string;		# index for associated SSL_sessionID
	handshake_cipher: string;	# agreed-upon cipher for session/conn.
};

# SSL_sessionID index - used to track version assosciated with a session id.
type SSL_sessionID_record: record {
	num_reuse: count;
	id: SSL_sessionID;			# literal session ID

	# everything below is an example of session vs connection monitoring.
	version: string;		# version assosciated with session id
	client_cert: X509;
	server_cert: X509;
	handshake_cipher: string;
};

global ssl_connections: table[conn_id] of ssl_connection_info;
global ssl_sessionIDs: table[string] of SSL_sessionID_record;
global ssl_connection_id = 0;

# Used when there's no issuer/subject/cipher.
const NONE = "<none>";

# --- SSL helper functions ---------
function new_ssl_connection(c: connection)
	{
	local conn = c$id;
	local new_id = ++ssl_connection_id;

	local info: ssl_connection_info;
	info$id = new_id;
	info$id_index = md5_hash(info$id);
	info$version = "";
	info$client_cert$issuer = NONE;
	info$client_cert$subject = NONE;
	info$server_cert$issuer = NONE;
	info$server_cert$subject = NONE;
	info$handshake_cipher = NONE;
	info$connection_id = conn;

	ssl_connections[conn] = info;
	c$addl = fmt("#%d", new_id);
	print ssl_log, fmt("%.6f #%d %s start",
				network_time(), new_id, id_string(conn));
	}

function new_sessionID_record(session: SSL_sessionID)
	{
	local info: SSL_sessionID_record;

	info$num_reuse = 1;
	info$client_cert$issuer = NONE;
	info$client_cert$subject = NONE;
	info$server_cert$issuer = NONE;
	info$server_cert$subject = NONE;
	info$handshake_cipher = NONE;
	local index = md5_hash(session);

	ssl_sessionIDs[index] = info;
	}

function ssl_get_cipher_name(cipherSuite: count): string
	{
	return cipherSuite in ssl_cipher_desc ?
		ssl_cipher_desc[cipherSuite] : "UNKNOWN";
	}

function ssl_get_version_string(version: int): string
	{
	if ( version == SSLv20 )
		return "2.0";
	else if ( version == SSLv30 )
		return "3.0";
	else if ( version == SSLv31 )
		return "3.1";
	else
		return "?.?";
	}

function ssl_con2str(c: connection): string
	{
	return fmt("%s:%s -> %s:%s",
			c$id$orig_h, c$id$orig_p, c$id$resp_h, c$id$resp_p);
	}

function lookup_ssl_conn(c: connection, func: string, log_if_new: bool)
	{
	if ( c$id !in ssl_connections )
		{
		if ( log_if_new )
			print ssl_log,
				fmt("creating new SSL connection in %s", func);
		new_ssl_connection(c);
		}
	}

event ssl_conn_weak(name: string, c: connection)
	{
	lookup_ssl_conn(c, "ssl_conn_weak", T);
	print ssl_log, fmt("%.6f #%d %s",
		network_time(), ssl_connections[c$id]$id, name);
	}

# --- SSL events -------------------

event ssl_ciphersuite_seen(c: connection, version: int,
			cipherSuites: cipher_suites_list, handshakeType: int)
	{
	lookup_ssl_conn(c, "ssl_ciphersuite_seen", T);

	local conn = ssl_connections[c$id];
	local version_string = ssl_get_version_string(version);
	local ht = "";

	if ( handshakeType == SSL_CLIENT_HELLO )
		ht = "CLIENT-HELLO";
	else if ( handshakeType == SSL_SERVER_HELLO )
		ht = "SERVER-HELLO";
	else if ( handshakeType == SSL_CLIENT_MASTER_KEY )
		ht = "CLIENT-MASTER-KEY";

	print ssl_log, fmt("%.6f #%d %s (%s)",
				network_time(), conn$id, ht, version_string);

	local msg = fmt("%.6f #%d cipher suites: ", network_time(), conn$id);

	# Set values for connection.  See if this is the first session conn.
	if ( handshakeType == SSL_SERVER_HELLO )
		{
		local handshake_cipher = ssl_get_cipher_name(cipherSuites[1]);
		conn$handshake_cipher = handshake_cipher;

		# if ( ssl_sessionIDs[conn$id_index]$handshake_cipher == NONE )
		#	ssl_sessionIDs[conn$id_index]$handshake_cipher = handshake_cipher;
		}

	for ( i in cipherSuites )
		{ # display a list of the cipher suites
		msg = fmt("%s %s (0x%x),", msg,
			ssl_get_cipher_name(cipherSuites[i]), cipherSuites[i]);

		# Demo: report clients who support weak ciphers.
		# if ( handshakeType == 0 && cipherSuites[i] in myWeakCiphers )
		#	{
		#	event ssl_conn_weak(fmt("SSL client supports weak cipher: %s (0x%x)",
		#		ssl_get_cipher_name(cipherSuites[i]), cipherSuites[i]), c );
		#	}

		# Demo: report servers who support weak ciphers.
		if ( version == SSLv20 && handshakeType == 1 &&
		     cipherSuites[i] in myWeakCiphers )
			{
			event ssl_conn_weak(fmt("SSLv2 server supports weak cipher: %s (0x%x)",
				ssl_get_cipher_name(cipherSuites[i]), cipherSuites[i]), c );
			}

		# Demo: report unknown ciphers.
		# if ( !(cipherSuites[i] in ssl_cipher_desc) )
		#	{
		#	event ssl_conn_weak(fmt("SSL: unknown cipher-spec: %s (0x%x)",
		#		ssl_get_cipher_name(cipherSuites[i]), cipherSuites[i]), c);
		#	}
		}

	print ssl_log, msg;
	}

event ssl_certificate_seen(c: connection, is_server: bool)
	{
	# Called whenever there's an certificate to analyze.
	# we could do something here, like...

	# if ( c$id$orig_h in hostsToIgnore )
	#	{
	#	ssl_store_certificates = F;
	#	ssl_verify_certificates = F;
	#	}
	# else
	#	{
	#	ssl_store_certificates = T;
	#	ssl_verify_certificates = T;
	#	}
	}

event ssl_certificate(c: connection, cert: X509, is_server: bool)
	{
	local direction = is_local_addr(c$id$orig_h) ? "client" : "server";

	lookup_ssl_conn(c, "ssl_certificate", T);
	local conn = ssl_connections[c$id];

	if( direction == "client" )
		conn$client_cert = cert;
	else
		{
		conn$server_cert = cert;

		# We have not filled in the field for the master session
		# for this connection.  Do it now.
		if ( ssl_sessionIDs[conn$id_index]$server_cert$subject == NONE )
			ssl_sessionIDs[conn$id_index]$server_cert$subject = cert$subject;
		}

	print ssl_log, fmt("%.6f #%d X.509 %s issuer %s",
			network_time(), conn$id, direction, cert$issuer);

	print ssl_log, fmt("%.6f #%d X.509 %s subject %s",
			network_time(), conn$id, direction, cert$subject);
	}

event ssl_conn_attempt(c: connection, version: int)
	{
	lookup_ssl_conn(c, "ssl_conn_attempt", F);
	local conn = ssl_connections[c$id];
	local version_string = ssl_get_version_string(version);
	print ssl_log, fmt("%.6f #%d SSL connection attempt version %s",
				network_time(), conn$id, version_string);

	conn$version = version_string;
	}

event ssl_conn_server_reply(c: connection, version: int)
	{
	lookup_ssl_conn(c, "ssl_conn_server_reply", T);
	local conn = ssl_connections[c$id];
	local version_string = ssl_get_version_string(version);

	print ssl_log, fmt("%.6f #%d SSL connection server reply, version %s",
				network_time(), conn$id, version_string);

	conn$version = version_string;
	}

event ssl_conn_established(c: connection, version: int, cipher_suite: count)
	{
	lookup_ssl_conn(c, "ssl_conn_established", T);
	local conn = ssl_connections[c$id];
	local version_string = ssl_get_version_string(version);

	print ssl_log,
		fmt("%.6f #%d handshake finished, version %s",
			network_time(), conn$id, version_string);

	if ( cipher_suite in myWeakCiphers )
		{
		event ssl_conn_weak(fmt("%.6f #%d weak cipher: %s (0x%x)",
			network_time(), conn$id,
			ssl_get_cipher_name(cipher_suite), cipher_suite), c);
		}

	++SSL_cipherCount[cipher_suite];

	# This should be the version identified with the session, unless
	# there is some renegotiation.  That will be caught later.
	conn$version = version_string;
	}

event process_X509_extensions(c: connection, ex: X509_extension)
	{
	lookup_ssl_conn(c, "process_X509_extensions", T);
	local conn = ssl_connections[c$id];

	local msg = fmt("%.6f #%d X.509 extensions: ", network_time(), conn$id);
	for ( i in ex )
	       msg = fmt("%s, %s", msg, ex[i]);

	print ssl_log, msg;
	}

event ssl_session_insertion(c: connection, id: SSL_sessionID)
	{
	local idd = c$id;

	if ( idd !in ssl_connections)
		{
		print ssl_log, "creating new SSL connection in ssl_session_insertion";
		new_ssl_connection(c);

		# None of the conn$object values will exist, so we leave this
		# to prevent needless crashing.
		return;
		}

	local conn = ssl_connections[idd];
	local id_index = md5_hash(id);

	# If there is no session with thIS id we create (a typical) one,
	# otherwise we move on.
	if ( id_index !in ssl_sessionIDs )
		{
		new_sessionID_record(id);
		local session = ssl_sessionIDs[id_index];
		session$version = conn$version;
		session$client_cert$subject = conn$client_cert$subject;
		session$server_cert$subject = conn$server_cert$subject;
		session$handshake_cipher = conn$handshake_cipher;
		session$id = id;
		conn$id_index = id_index;
		}

	else
		{ # should we ever get here?
		session = ssl_sessionIDs[id_index];
		conn$id_index = id_index;
		}
	}

event ssl_conn_reused(c: connection, session_id: SSL_sessionID)
	{
	lookup_ssl_conn(c, "ssl_conn_reused", T);
	local conn = ssl_connections[c$id];
	local id_index = md5_hash(session_id);

	if ( id_index !in ssl_sessionIDs )
		{
		new_sessionID_record(session_id);
		local session = ssl_sessionIDs[id_index];
		session$version = conn$version;
		session$client_cert$subject = conn$client_cert$subject;
		session$server_cert$subject = conn$server_cert$subject;
		session$id = session_id;
		}
	else
		session = ssl_sessionIDs[id_index];

	print ssl_log, fmt("%.6f #%d reusing former SSL session: %s",
				network_time(), conn$id, id_index);

	++session$num_reuse;

	# At this point, the connection values have been set.  We can then
	# compare session and connection values with some confidence.
	if ( session$version != conn$version ||
	     session$handshake_cipher != conn$handshake_cipher )
		{
		NOTICE([$note=SSL_SessConIncon, $conn=c,
			$msg="session violation"]);
		++c$hot;
		}
	}

event ssl_X509_error(c: connection, err: int, err_string: string)
	{
	if ( err in x509_ignore_errors )
		return;

	lookup_ssl_conn(c, "ssl_X509_error", T);
	local conn = ssl_connections[c$id];
	local error =
		err in x509_errors ?  x509_errors[err] : "unknown X.509 error";

	local h_flag = " ";
	if ( err in x509_hot_errors )
		{
		NOTICE([$note=SSL_X509Violation, $conn=c, $msg=error]);
		++c$hot;
		h_flag = "hot";
		}

	print ssl_log,
		fmt("%.6f #%d X.509 %s error %s (%s)",
			network_time(), conn$id, h_flag, error, err_string);
	}

event connection_state_remove(c: connection)
	{
	delete ssl_connections[c$id];
	}

event bro_done()
	{
	print ssl_log, "Cipher suite statistics: ";
	for ( i in SSL_cipherCount )
		print ssl_log, fmt("%s (0x%x): %d", ssl_get_cipher_name(i), i,
					SSL_cipherCount[i]);

	print ssl_log, ("count     session ID");
	print ssl_log, ("-----     ---------------------------------");
	for ( j in ssl_sessionIDs )
		if ( ssl_sessionIDs[j]$server_cert$subject != NONE )
			{
			print ssl_log,
				fmt("(%s)      %s   %s",
					ssl_sessionIDs[j]$num_reuse,
					ssl_sessionIDs[j]$server_cert$subject,
					j);
			}
	}
