// $Id: Conn.cc,v 1.8 2005/03/17 09:22:17 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2000, 2001, 2002
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include "config.h"

#include <ctype.h>

#include "Net.h"
#include "NetVar.h"
#include "Conn.h"
#include "Event.h"
#include "Sessions.h"
#include "Logger.h"
#include "Timer.h"

HashKey* ConnID::BuildConnKey() const
	{
	Key key;

	// Lookup up connection based on canonical ordering, which is
	// the smaller of <src addr, src port> and <dst addr, dst port>
	// followed by the other.
	if ( is_one_way ||
	     addr_port_canon_lt(src_addr, src_port, dst_addr, dst_port) )
		{
		copy_addr(src_addr, key.ip1);
		copy_addr(dst_addr, key.ip2);
		key.port1 = src_port;
		key.port2 = dst_port;
		}
	else
		{
		copy_addr(dst_addr, key.ip1);
		copy_addr(src_addr, key.ip2);
		key.port1 = dst_port;
		key.port2 = src_port;
		}

	return new HashKey(&key, sizeof(key));
	}

void ConnectionTimer::Init(Connection* arg_conn, timer_func arg_timer,
				int arg_do_expire)
	{
	conn = arg_conn;
	timer = arg_timer;
	do_expire = arg_do_expire;
	Ref(conn);
	}

ConnectionTimer::~ConnectionTimer()
	{
	if ( conn->RefCnt() < 1 )
		internal_error("reference count inconsistency in ~ConnectionTimer");

	conn->RemoveTimer(this);
	Unref(conn);
	}

void ConnectionTimer::Dispatch(double t, int is_expire)
	{
	if ( is_expire && ! do_expire )
		return;

	// Remove ourselves from the connection's set of timers so
	// it doesn't try to cancel us.
	conn->RemoveTimer(this);

	(conn->*timer)(t);

	if ( conn->RefCnt() < 1 )
		internal_error("reference count inconsistency in ConnectionTimer::Dispatch");
	}

// This is not very nice. But adding some mapping table isn't really
// better ...

#include "NTP.h"
#include "TCP.h"
#include "DNS.h"
#include "NetbiosSSN.h"
#include "RPC.h"

IMPLEMENT_SERIAL(ConnectionTimer, SER_CONNECTION_TIMER);

bool ConnectionTimer::DoSerialize(SerialInfo* info) const
	{
	DO_SERIALIZE(SER_CONNECTION_TIMER, Timer);

	// We enumerate all the possible timer functions here ... This
	// has to match the list is DoUnserialize()!
	char type = 0;

	if ( timer == timer_func(&Connection::DeleteTimer) )
		type = 1;
	else if ( timer == timer_func(&Connection::InactivityTimer) )
		type = 2;
	else if ( timer == timer_func(&Connection::StatusUpdateTimer) )
		type = 3;
	else if ( timer == timer_func(&NTP_Session::ExpireTimer) )
		type = 4;
	else if ( timer == timer_func(&TCP_Connection::AttemptTimer) )
		type = 5;
	else if ( timer == timer_func(&TCP_Connection::DeleteTimer) )
		type = 6;
	else if ( timer == timer_func(&TCP_Connection::ExpireTimer) )
		type = 7;
	else if ( timer == timer_func(&TCP_Connection::PartialCloseTimer) )
		type = 8;
	else if ( timer == timer_func(&TCP_Connection::ResetTimer) )
		type = 9;
	else if ( timer == timer_func(&UDP_DNS::ExpireTimer) )
		type = 10;
	else if ( timer == timer_func(&UDP_NetbiosSSN::ExpireTimer) )
		type = 11;
	else if ( timer == timer_func(&UDP_RPC::ExpireTimer) )
		type = 12;
	else
		internal_error("unknown function in ConnectionTimer::DoSerialize()");

	return conn->Serialize(info) && SERIALIZE(type) && SERIALIZE(do_expire);

	}

bool ConnectionTimer::DoUnserialize(UnserialInfo* info)
	{
	DO_UNSERIALIZE(Timer);

	conn = Connection::Unserialize(info);
	if ( ! conn )
		return false;

	char type;

	if ( ! UNSERIALIZE(&type) || ! UNSERIALIZE(&do_expire) )
		return false;

	switch ( type ) {
	case 1:
		timer = timer_func(&Connection::DeleteTimer);
		break;
	case 2:
		timer = timer_func(&Connection::InactivityTimer);
		break;
	case 3:
		timer = timer_func(&Connection::StatusUpdateTimer);
		break;
	case 4:
		timer = timer_func(&NTP_Session::ExpireTimer);
		break;
	case 5:
		timer = timer_func(&TCP_Connection::AttemptTimer);
		break;
	case 6:
		timer = timer_func(&TCP_Connection::DeleteTimer);
		break;
	case 7:
		timer = timer_func(&TCP_Connection::ExpireTimer);
		break;
	case 8:
		timer = timer_func(&TCP_Connection::PartialCloseTimer);
		break;
	case 9:
		timer = timer_func(&TCP_Connection::ResetTimer);
		break;
	case 10:
		timer = timer_func(&UDP_DNS::ExpireTimer);
		break;
	case 11:
		timer = timer_func(&UDP_NetbiosSSN::ExpireTimer);
		break;
	case 12:
		timer = timer_func(&UDP_RPC::ExpireTimer);
		break;
	default:
		info->s->Error("unknown connection timer function");
		return false;
	}

	return true;
	}

unsigned int Connection::total_connections = 0;
unsigned int Connection::current_connections = 0;

Connection::Connection(NetSessions* s, HashKey* k, double t, const ConnID* id)
	{
	sessions = s;
	key = k;
	start_time = last_time = t;

	copy_addr(id->src_addr, orig_addr);
	copy_addr(id->dst_addr, resp_addr);
	orig_port = id->src_port;
	resp_port = id->dst_port;

	conn_val = 0;
	orig_endp = resp_endp = 0;
	login_conn = 0;

	is_active = 1;
	skip = 0;
	weird = 0;
	persistent = 0;

	suppress_event = 0;

	record_contents = record_packets = 1;

	orig_match_state = resp_match_state = 0;

	timers_canceled = 0;
	inactivity_timeout = 0;
	installed_status_timer = 0;

	finished = 0;

	++current_connections;
	++total_connections;
	}

Connection::~Connection()
	{
	if ( ! finished )
		internal_error("Done() not called before destruction of Connection");

	CancelTimers();

	if ( conn_val )
		{
		conn_val->SetOrigin(0);
		Unref(conn_val);
		}

	delete key;

	delete orig_match_state;
	delete resp_match_state;

	--current_connections;
	}

void Connection::Done()
	{
	finished = 1;
	}

void Connection::SetContentsFile(unsigned int /* direction */, BroFile* /* f */)
	{
	RunTime("connection type does not support writing to a contents file");
	}

BroFile* Connection::GetContentsFile(unsigned int /* direction */) const
	{
	RunTime("connection type does not support writing to a contents file");
	return 0;
	}

void Connection::SetLifetime(double lifetime)
	{
	ADD_TIMER(&Connection::DeleteTimer, network_time + lifetime, 0,
			TIMER_CONN_DELETE);
	}

void Connection::DeleteTimer(double /* t */)
	{
	if ( is_active )
		Event(connection_timeout);

	sessions->Remove(this);
	}

int killed_by_inactivity = 0;

void Connection::InactivityTimer(double t)
	{
	// If the inactivity_timeout is zero, there has been an active
	// timeout once, but it's disabled now. We do nothing then.
	if ( inactivity_timeout )
		{
		if ( last_time + inactivity_timeout <= t )
			{
			Event(connection_timeout);
			sessions->Remove(this);
			++killed_by_inactivity;
			}
		else
			ADD_TIMER(&Connection::InactivityTimer,
					last_time + inactivity_timeout, 0,
					TIMER_CONN_INACTIVITY);
		}
	}

void Connection::SetInactivityTimeout(double timeout)
	{
	// Check if we are inactive for the given amount of time already.
	if ( timeout && last_time + timeout < network_time )
		{
		Event(connection_timeout);
		sessions->Remove(this);
		++killed_by_inactivity;
		}

	// We add a new inactivity timer even if there already is one.  When
	// it fires, we always use the current value to check for inactivity.
	if ( timeout )
		ADD_TIMER(&Connection::InactivityTimer,
				last_time + timeout, 0, TIMER_CONN_INACTIVITY);

	inactivity_timeout = timeout;
	}

void Connection::EnableStatusUpdateTimer()
	{
	if ( connection_status_update && connection_status_update_interval )
		{
		ADD_TIMER(&Connection::StatusUpdateTimer,
			network_time + connection_status_update_interval, 0,
			TIMER_CONN_STATUS_UPDATE);
		installed_status_timer = 1;
		}
	}

void Connection::StatusUpdateTimer(double t)
	{
	val_list* vl = new val_list(1);
	vl->append(BuildConnVal());
	ConnectionEvent(connection_status_update, vl);
	ADD_TIMER(&Connection::StatusUpdateTimer,
			network_time + connection_status_update_interval, 0,
			TIMER_CONN_STATUS_UPDATE);
	}

RecordVal* Connection::BuildConnVal()
	{
	if ( ! conn_val )
		{
		conn_val = new RecordVal(connection_type);

		TransportProto prot_type = ConnTransport();

		RecordVal* id_val = new RecordVal(conn_id);
		id_val->Assign(0, new AddrVal(orig_addr));
		id_val->Assign(1, new PortVal(ntohs(orig_port), prot_type));
		id_val->Assign(2, new AddrVal(resp_addr));
		id_val->Assign(3, new PortVal(ntohs(resp_port), prot_type));
		conn_val->Assign(0, id_val);

		orig_endp = new RecordVal(endpoint);
		orig_endp->Assign(0, new Val(0, TYPE_COUNT));
		orig_endp->Assign(1, new Val(0, TYPE_COUNT));
		conn_val->Assign(1, orig_endp);

		resp_endp = new RecordVal(endpoint);
		resp_endp->Assign(0, new Val(0, TYPE_COUNT));
		resp_endp->Assign(1, new Val(0, TYPE_COUNT));
		conn_val->Assign(2, resp_endp);

		// conn_val->Assign(3, new Val(start_time, TYPE_TIME));	// ###
		conn_val->Assign(5, new StringVal(""));	// service
		conn_val->Assign(6, new StringVal(""));	// addl
		conn_val->Assign(7, new Val(0, TYPE_COUNT));	// hot
		}

	UpdateEndpointVal(orig_endp, 1);
	UpdateEndpointVal(resp_endp, 0);

	conn_val->Assign(3, new Val(start_time, TYPE_TIME));	// ###
	conn_val->Assign(4, new Val(last_time - start_time, TYPE_INTERVAL));

	conn_val->SetOrigin(this);

	Ref(conn_val);

	return conn_val;
	}

// Returns true if the character at s separates a version number.
static inline bool is_version_sep(const char* s, const char* end)
	{
	return
		// foo-1.2.3
			(s < end - 1 && ispunct(s[0]) && isdigit(s[1])) ||
		// foo-v1.2.3
			(s < end - 2 && ispunct(s[0]) &&
			 tolower(s[1]) == 'v' && isdigit(s[2])) ||
		// foo 1.2.3
			isspace(s[0]);
	}

Val* Connection::BuildVersionVal(const char* s, int len)
	{
	Val* name = 0;
	Val* major = 0;
	Val* minor = 0;
	Val* minor2 = 0;
	Val* addl = 0;

	const char* last = s + len;
	const char* e = s;

	// This is all just a guess...

	// Eat non-alpha-numerical chars.
	for ( ; s < last && ! isalnum(*s); ++s )
		;

	// Leading characters are the program name.
	// (first character must not be a digit)
	if ( isalpha(*s) )
		{
		for ( e = s; e < last && ! is_version_sep(e, last); ++e )
			;

		if ( s != e )
			name = new StringVal(e - s, s);
		}

	// Find first number - that's the major version.
	for ( s = e; s < last && ! isdigit(*s); ++s )
		;
	for ( e = s; e < last && isdigit(*e); ++e )
		;

	if ( s != e )
		major = new Val(atoi(s), TYPE_INT);

	// Find second number seperated only by punctuation chars -
	// that's the minor version.
	for ( s = e; s < last && ispunct(*s); ++s )
		;
	for ( e = s; e < last && isdigit(*e); ++e )
		;

	if ( s != e )
		minor = new Val(atoi(s), TYPE_INT);

	// Find second number seperated only by punctuation chars; -
	// that's the minor version.
	for ( s = e; s < last && ispunct(*s); ++s )
		;
	for ( e = s; e < last && isdigit(*e); ++e )
		;

	if ( s != e )
		minor2 = new Val(atoi(s), TYPE_INT);

	// Anything after following punctuation and until next white space is
	// an additional version string.
	for ( s = e; s < last && ispunct(*s); ++s )
		;
	for ( e = s; e < last && ! isspace(*e); ++e )
		;

	if ( s != e )
		addl = new StringVal(e - s, s);

	// If we do not have a name yet, the next alphanumerical string is it.
	if ( ! name )
		{ // eat non-alpha-numerical characters
		for ( s = e; s < last && ! isalpha(*s); ++s )
			;

		// Get name.
		for ( e = s; e < last && (isalnum(*e) || *e == '_'); ++e )
			;

		if ( s != e )
			name = new StringVal(e - s, s);
		}

	// We need at least a name.
	if ( ! name )
		return 0;

	RecordVal* version = new RecordVal(software_version);
	version->Assign(0, major ? major : new Val(-1, TYPE_INT));
	version->Assign(1, minor ? minor : new Val(-1, TYPE_INT));
	version->Assign(2, minor2 ? minor2 : new Val(-1, TYPE_INT));
	version->Assign(3, addl ? addl : new StringVal(""));

	RecordVal* sw = new RecordVal(software);
	sw->Assign(0, name);
	sw->Assign(1, version);

	return sw;
	}

int Connection::VersionFoundEvent(const uint32* addr, const char* s, int len)
	{
	if ( ! software_version_found )
		return 1;

	if ( ! is_printable(s, len) )
		return 0;

	Val* val = BuildVersionVal(s, len);
	if ( ! val )
		{
		val_list* vl = new val_list;
		vl->append(BuildConnVal());
		vl->append(new AddrVal(addr));
		vl->append(new StringVal(len, s));
		ConnectionEvent(software_parse_error, vl);
		return 0;
		}

	val_list* vl = new val_list;
	vl->append(BuildConnVal());
	vl->append(new AddrVal(addr));
	vl->append(val);
	vl->append(new StringVal(len, s));
	ConnectionEvent(software_version_found, vl);

	return 1;
	}

int Connection::UnparsedVersionFoundEvent(const uint32* addr,
					const char* full, int len)
	{
	if ( ! software_unparsed_version_found )
		return 1;

	// Skip leading white space.
	while ( len && isspace(*full) )
		{
		--len;
		++full;
		}

	if ( ! is_printable(full, len) )
		return 0;

	val_list* vl = new val_list;
	vl->append(BuildConnVal());
	vl->append(new AddrVal(addr));
	vl->append(new StringVal(len, full));
	ConnectionEvent(software_unparsed_version_found, vl);

	return 1;
	}

void Connection::Event(EventHandlerPtr f, const char* name)
	{
	if ( ! f )
		return;

	val_list* vl = new val_list(2);
	if ( name )
		vl->append(new StringVal(name));
	vl->append(BuildConnVal());

	ConnectionEvent(f, vl);
	}

void Connection::Event(EventHandlerPtr f, Val* v1, Val* v2)
	{
	if ( ! f )
		{
		Unref(v1);
		Unref(v2);
		return;
		}

	val_list* vl = new val_list(3);
	vl->append(BuildConnVal());
	vl->append(v1);

	if ( v2 )
		vl->append(v2);

	ConnectionEvent(f, vl);
	}

void Connection::ConnectionEvent(EventHandlerPtr f, val_list* vl)
	{
	if ( ! f )
		{
		// This may actually happen if there is no local handler
		// and a previously existing remote handler went away.
		loop_over_list(*vl, i)
			Unref((*vl)[i]);
		delete vl;
		return;
		}

	// "this" is passed as a cookie for the event
	mgr.QueueEvent(f, vl, SOURCE_LOCAL, this);
	}

void Connection::Weird(const char* name)
	{
	++weird;
	if ( conn_weird )
		Event(conn_weird, name);
	else
		fprintf(stderr, "%.06f weird: %s\n", network_time, name);
	}

void Connection::Weird(const char* name, const char* addl)
	{
	++weird;
	if ( conn_weird_addl )
		{
		val_list* vl = new val_list(3);

		vl->append(new StringVal(name));
		vl->append(BuildConnVal());
		vl->append(new StringVal(addl));

		ConnectionEvent(conn_weird_addl, vl);
		}

	else
		fprintf(stderr, "%.06f weird: %s (%s)\n", network_time, name, addl);
	}

void Connection::Weird(const char* name, int addl_len, const char* addl)
	{
	++weird;
	if ( conn_weird_addl )
		{
		val_list* vl = new val_list(3);

		vl->append(new StringVal(name));
		vl->append(BuildConnVal());
		vl->append(new StringVal(addl_len, addl));

		ConnectionEvent(conn_weird_addl, vl);
		}

	else
		{
		fprintf(stderr, "%.06f weird: %s (", network_time, name);
		for ( int i = 0; i < addl_len; ++i )
			fputc(addl[i], stderr);
		fprintf(stderr, ")\n");
		}
	}

void Connection::AddTimer(timer_func timer, double delta_t, int do_expire,
		TimerType type)
	{
	Timer* conn_timer = new ConnectionTimer(this, timer, delta_t, do_expire,
		type);
	timer_mgr->Add(conn_timer);
	timers.append(conn_timer);
	}

void Connection::RemoveTimer(Timer* t)
	{
	timers.remove(t);
	}

void Connection::CancelTimers()
	{
	loop_over_list(timers, i)
		timer_mgr->Cancel(timers[i]);

	timers_canceled = 1;
	timers.clear();
	}

void Connection::InitEndpointMatcher(const IP_Hdr* ip, int caplen, int from_orig)
	{
	if ( ! rule_matcher )
		return;

	if ( from_orig )
		{
		if ( orig_match_state )
			{
			rule_matcher->FinishEndpoint(orig_match_state);
			delete orig_match_state;
			}

		orig_match_state = rule_matcher->InitEndpoint(this, ip, caplen, resp_match_state);
		}

	else
		{
		if ( resp_match_state )
			{
			rule_matcher->FinishEndpoint( resp_match_state );
			delete resp_match_state;
			}

		resp_match_state = rule_matcher->InitEndpoint(this, ip, caplen, orig_match_state);
		}
	}

void Connection::FinishEndpointMatcher()
	{
	if ( ! rule_matcher )
		return;

	if ( orig_match_state )
		rule_matcher->FinishEndpoint(orig_match_state);

	if ( resp_match_state )
		rule_matcher->FinishEndpoint(resp_match_state);

	delete orig_match_state;
	delete resp_match_state;

	orig_match_state = resp_match_state = 0;
	}

unsigned int Connection::MemoryAllocation() const
	{
	return padded_sizeof(*this)
		+ (key ? key->MemoryAllocation() : 0)
		+ (timers.MemoryAllocation() - padded_sizeof(timers))
		+ (conn_val ? conn_val->MemoryAllocation() : 0)
		// FIXME: Implement MemoryAllocation() for these.
		+ (orig_match_state ? padded_sizeof(*orig_match_state) : 0)
		+ (resp_match_state ? padded_sizeof(*resp_match_state) : 0)
		// login_conn is just a casted 'this'.
		;
	}

unsigned int Connection::MemoryAllocationConnVal() const
	{
	return conn_val ? conn_val->MemoryAllocation() : 0;
	}

bool Connection::Serialize(SerialInfo* info) const
	{
	return SerialObj::Serialize(info);
	}

Connection* Connection::Unserialize(UnserialInfo* info)
	{
	return (Connection*) SerialObj::Unserialize(info, SER_CONNECTION);
	}

bool Connection::DoSerialize(SerialInfo* info) const
	{
	DO_SERIALIZE(SER_CONNECTION, BroObj);

	// First we write the members which are needed to
	// create the HashKey.
	for ( int j = 0; j < NUM_ADDR_WORDS; ++j )
		if ( ! SERIALIZE(orig_addr[j]) || ! SERIALIZE(resp_addr[j]) )
			return false;

	if ( ! SERIALIZE(orig_port) || ! SERIALIZE(resp_port) )
		return false;

	if ( ! SERIALIZE(timers.length()) )
		return false;

	loop_over_list(timers, i)
		if ( ! timers[i]->Serialize(info) )
			return false;

	SERIALIZE_OPTIONAL(conn_val);
	SERIALIZE_OPTIONAL(orig_endp);
	SERIALIZE_OPTIONAL(resp_endp);

	// FIXME: RuleEndpointState not yet serializable

	return SERIALIZE(start_time) &&
		SERIALIZE(last_time) &&
		SERIALIZE(inactivity_timeout) &&
		SERIALIZE(suppress_event) &&
		SERIALIZE(login_conn != 0) &&
		SERIALIZE_BIT(installed_status_timer) &&
		SERIALIZE_BIT(timers_canceled) &&
		SERIALIZE_BIT(is_active) &&
		SERIALIZE_BIT(skip) &&
		SERIALIZE_BIT(weird) &&
		SERIALIZE_BIT(finished) &&
		SERIALIZE_BIT(record_packets) &&
		SERIALIZE_BIT(record_contents) &&
		SERIALIZE_BIT(persistent);
	}

bool Connection::DoUnserialize(UnserialInfo* info)
	{
	// Make sure this is initialized for the condition in Unserialize().
	persistent = 0;

	DO_UNSERIALIZE(BroObj);

	// Build the hash key first. Some of the recursive *::Unserialize()
	// functions may need it.
	for ( int i = 0; i < NUM_ADDR_WORDS; ++i )
		if ( ! UNSERIALIZE(&orig_addr[i]) || ! UNSERIALIZE(&resp_addr[i]) )
			goto error;

	if ( ! UNSERIALIZE(&orig_port) || ! UNSERIALIZE(&resp_port) )
		goto error;

	ConnID id;
	id.src_addr = orig_addr;
	id.dst_addr = resp_addr;
	// This doesn't work for ICMP. But I guess this is not really important.
	id.src_port = orig_port;
	id.dst_port = resp_port;
	id.is_one_way = 0;	// ### incorrect for ICMP
	key = id.BuildConnKey();

	int len;
	if ( ! UNSERIALIZE(&len) )
		goto error;

	while ( len-- )
		{
		Timer* t = Timer::Unserialize(info);
		if ( ! t )
			goto error;
		timers.append(t);
		}

	UNSERIALIZE_OPTIONAL(conn_val,
			(RecordVal*) Val::Unserialize(info, connection_type));
	UNSERIALIZE_OPTIONAL(orig_endp,
			(RecordVal*) Val::Unserialize(info, endpoint));
	UNSERIALIZE_OPTIONAL(resp_endp,
			(RecordVal*) Val::Unserialize(info, endpoint));

	if ( ! (UNSERIALIZE(&start_time) &&
		UNSERIALIZE(&last_time) &&
		UNSERIALIZE(&inactivity_timeout) &&
		UNSERIALIZE(&suppress_event)) )
		goto error;

	bool has_login_conn;
	if ( ! UNSERIALIZE(&has_login_conn) )
		goto error;

	login_conn = has_login_conn ? (LoginConn*) this : 0;

	UNSERIALIZE_BIT(installed_status_timer);
	UNSERIALIZE_BIT(timers_canceled);
	UNSERIALIZE_BIT(is_active);
	UNSERIALIZE_BIT(skip);
	UNSERIALIZE_BIT(weird);
	UNSERIALIZE_BIT(finished);
	UNSERIALIZE_BIT(record_packets);
	UNSERIALIZE_BIT(record_contents);
	UNSERIALIZE_BIT(persistent);

	// Hmm... Why does each connection store a sessions ptr?
	sessions = ::sessions;

	orig_match_state = resp_match_state = 0;

	return true;

error:
	abort();
	CancelTimers();
	return false;
	}
