// $Id: File.h,v 1.4 2005/01/23 22:41:25 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2002
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef file_h
#define file_h

#include <fcntl.h>
#include "util.h"
#include "Obj.h"
#include "Attr.h"

#ifdef USE_OPENSSL
# ifdef NEED_KRB5_H
#  include <krb5.h>
# endif // NEED_KRB5_H
extern "C" {
# include "openssl/evp.h"
# include "openssl/pem.h"
# include "openssl/err.h"
}
#endif

class BroType;
class RotateTimer;

class BroFile : public BroObj {
public:
	BroFile(FILE* arg_f);
	BroFile(FILE* arg_f, const char* filename, const char* access);
	BroFile(const char* filename, const char* access, BroType* arg_t = 0);
	virtual ~BroFile();

	// Returns false if an error occured.
	int Write(const char* data, int len = 0);

	void Flush()	{ fflush(f); }

	FILE* Seek(long position);	// seek to absolute position

	void SetBuf(bool buffered);	// false=line buffered, true=fully buffered

	const char* Name()	{ return name; }

	BroType* FType() const	{ return t; }

	// Whether the file is open in a general sense; it might
	// not be open as a Unix file due to our management of
	// a finite number of FDs.
	int IsOpen() const	{ return is_open; }

	// Returns true if the close made sense, false if it was already
	// closed, not active, or whatever.
	int Close();

	void Describe(ODesc* d) const;

	// Rotates the logfile. Returns rotate_info.
	RecordVal* Rotate();

	// Set &rotate_interval, &rotate_size, and &postprocessor attributes.
	void SetAttrs(Attributes* attrs);

	// Set rotate/postprocessor for all files that don't define them
	// by their own. (interval/max_size=0 for no rotation; size in bytes).
	static void SetDefaultRotation(double interval, double max_size);

	// Close all files which are managed by us.
	static void CloseCachedFiles();

	bool Serialize(SerialInfo* info) const;
	static BroFile* Unserialize(UnserialInfo* info);

protected:
	friend class RotateTimer;

	BroFile()	{ Init(); }
	void Init();
	bool Open(FILE* f = 0);	// if file is given, it's an open file to use

	BroFile* Prev()	{ return prev; }
	BroFile* Next()	{ return next; }
	void SetPrev(BroFile* f)	{ prev = f; }
	void SetNext(BroFile* f)	{ next = f; }

	void Suspend();
	void PurgeCache();
	void Unlink();
	void InsertAtBeginning();
	void MoveToBeginning();
	void InstallRotateTimer();

	// Returns nil if the file is not active, was in error, etc.
	// (Protected because we do not want anyone to write directly
	// to the file.)
	FILE* File();
	FILE* BringIntoCache();

	// Stats the file to get its current size.
	void UpdateFileSize();

	// Initialize encryption with the given public key.
	void InitEncrypt(const char* keyfile);
	// Finalize encryption.
	void FinishEncrypt();

	DECLARE_SERIAL(BroFile);

	FILE* f;
	BroType* t;
	char* name;
	char* access;
	int is_in_cache;	// whether it's currently in the open-file cache
	int is_open;	// whether the file is open in a general sense
	int okay_to_manage;	// we're allowed to cache/uncache
	long position;	// only valid if ! is_in_cache
	BroFile* next;	// doubly-linked list of cached files
	BroFile* prev;
	Attributes* attrs;
	double rotate_interval;

	// Sizes are double's so that it's easy to specify large
	// ones with scientific notation, and so they can exceed 4GB.
	double rotate_size;
	double current_size;

	Timer* rotate_timer;
	double open_time;
	bool dont_rotate; // See InstallRotateTimer()

#ifdef USE_OPENSSL
	EVP_PKEY* pub_key;
	EVP_CIPHER_CTX* cipher_ctx;

	static const int MIN_BUFFER_SIZE = 1024;
	unsigned char* cipher_buffer;
#endif

};

#endif
