// $Id: Gnutella.cc,v 1.3 2005/09/02 23:00:06 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2001, 2002
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include "config.h"

#include <ctype.h>

#include "NetVar.h"
#include "HTTP.h"
#include "Gnutella.h"
#include "Event.h"


GnutellaMsgState::GnutellaMsgState()
	{
	buffer = "";
	current_offset = 0;
	headers = "";
	msg_hops = 0;
	msg_len = 0;
	msg_pos = 0;
	msg_type = 0;
	msg_sent = 1;
	msg_ttl = 0;
	payload_left = 0;
	got_CR = 0;
	payload_len = 0;
	}


GnutellaConn::GnutellaConn(NetSessions* s, HashKey* k, double t, 
			    const ConnID* id, const struct tcphdr* tp)
: TCP_Connection(s, k, t, id, tp)
	{
	state = 0;
	new_state = 0;
	http_conn = 0;
	sent_establish = 0;

	orig_s = s;
	orig_k = k;
	orig_t = t;
	orig_id = id;
	orig_tp = tp;

	ms = 0;

	orig_msg_state = new GnutellaMsgState();
	resp_msg_state = new GnutellaMsgState();
	}

void GnutellaConn::BuildEndpoints()
	{
	orig->AddContentsProcessor(new TCP_Contents(orig));
	resp->AddContentsProcessor(new TCP_Contents(resp));
	}

void GnutellaConn::Done()
	{
	if ( http_conn )
		{
		http_conn->Done();
		TCP_Connection::Done();
		return;
		}

	if ( ! sent_establish && (gnutella_establish || gnutella_not_establish) )
		{
		val_list* vl = new val_list;

		vl->append(BuildConnVal());

		if ( Established() && gnutella_establish )
			ConnectionEvent(gnutella_establish, vl);
		else if ( ! Established () && gnutella_not_establish )
			ConnectionEvent(gnutella_not_establish, vl);
		}

	if ( gnutella_partial_binary_msg )
		{
		GnutellaMsgState* p = orig_msg_state;

		for ( int i = 0; i < 2; ++i, p = resp_msg_state )
			{
			if ( ! p->msg_sent && p->msg_pos )
				{
				val_list* vl = new val_list;

				vl->append(BuildConnVal());
				vl->append(new StringVal(p->msg));
				vl->append(new Val((i == 0), TYPE_BOOL));
				vl->append(new Val(p->msg_pos, TYPE_COUNT));

				ConnectionEvent(gnutella_partial_binary_msg, vl);
				}

			else if ( ! p->msg_sent && p->payload_left )
				SendEvents(p, (i == 0));
			}
		}

	TCP_Connection::Done();
	}


int GnutellaConn::NextLine(u_char* data, int len)
	{
	if ( ! ms )
		return 0;

	if ( Established() || ms->current_offset >= len )
		return 0;

	for ( ; ms->current_offset < len; ++ms->current_offset )
		{
		if ( data[ms->current_offset] == '\r' )
			ms->got_CR = 1;

		else if ( data[ms->current_offset] == '\n' && ms->got_CR )
			{
			ms->got_CR = 0;
			++ms->current_offset;
			return 1;
			}
		else
			ms->buffer += data[ms->current_offset];
		}

	return 0;
	}


int GnutellaConn::IsHTTP(string header)
	{
	if ( header.find(" HTTP/1.") == string::npos )
		return 0;

	if ( http_request || http_reply )
		{
		http_conn =
			new HTTP_Conn(orig_s, orig_k, orig_t, orig_id, orig_tp);
		http_conn->Init();
		}

	if ( gnutella_http_notify )
		{
		val_list* vl = new val_list;

		vl->append(BuildConnVal());
		ConnectionEvent(gnutella_http_notify, vl);
		}

	return 1;
	}


int GnutellaConn::GnutellaOK(string header)
	{
	if ( strncmp("GNUTELLA", header.data(), 8) )
		return 0;

	int codepos = header.find(' ') + 1;
	if ( ! strncmp("200", header.data() + codepos, 3) )
		return 1;

	return 0;
	}


void GnutellaConn::DeliverLines(TCP_Endpoint* s,
				 int /* seq */, int len, u_char* data)
	{
	if ( ! ms )
		return;

	while ( NextLine(data, len) )
		{
		if ( ms->buffer.length() )
			{
			if ( ms->headers.length() == 0 )
				{
				if ( IsHTTP(ms->buffer) )
					return;
				if ( GnutellaOK(ms->buffer) )
					new_state |=
						(s == orig) ? ORIG_OK : RESP_OK;
				}

			ms->headers = ms->headers + "\r\n" + ms->buffer;
			ms->buffer = "";
			}
		else
			{
			if ( gnutella_text_msg )
				{
				val_list* vl = new val_list;

				vl->append(BuildConnVal());
				vl->append(new Val((s == orig), TYPE_BOOL));
				vl->append(new StringVal(ms->headers.data()));

				ConnectionEvent(gnutella_text_msg, vl);
				}

			ms->headers = "";
			state |= new_state;

			if ( Established () && gnutella_establish )
				{
				val_list* vl = new val_list;

				sent_establish = 1;
				vl->append(BuildConnVal());

				ConnectionEvent(gnutella_establish, vl);
				}
			}
		}
	}


void GnutellaConn::DeliverHTTP(TCP_Endpoint* s, int seq, int len, u_char* data)
	{
	if ( http_conn )
		http_conn->Deliver(s, seq, len, data);
	}


void GnutellaConn::DissectMessage(char* msg)
	{
	if ( ! ms )
		return;

	ms->msg_type = msg[16];
	ms->msg_ttl = msg[17];
	ms->msg_hops = msg[18];

	memcpy(&ms->msg_len, &msg[19], 4);
	}


void GnutellaConn::SendEvents(GnutellaMsgState* p, bool is_orig)
	{
	if ( p->msg_sent )
		return;

	if ( gnutella_binary_msg )
		{
		val_list* vl = new val_list;

		vl->append(BuildConnVal());
		vl->append(new Val(is_orig, TYPE_BOOL));
		vl->append(new Val(p->msg_type, TYPE_COUNT));
		vl->append(new Val(p->msg_ttl, TYPE_COUNT));
		vl->append(new Val(p->msg_hops, TYPE_COUNT));
		vl->append(new Val(p->msg_len, TYPE_COUNT));
		vl->append(new StringVal(p->payload));
		vl->append(new Val(p->payload_len, TYPE_COUNT));
		vl->append(new Val((p->payload_len <
				    min(p->msg_len, GNUTELLA_MAX_PAYLOAD)),
				   TYPE_BOOL));
		vl->append(new Val((p->payload_left == 0), TYPE_BOOL));

		ConnectionEvent(gnutella_binary_msg, vl);
		}
	}


void GnutellaConn::DeliverMessages(TCP_Endpoint* s, int seq, 
				    int len, u_char* data)
	{
	if ( ! ms )
		return;

	while ( ms->current_offset < len )
		{
		ms->msg_sent = 0;

		unsigned int bytes_left = len - ms->current_offset;
		unsigned int needed = 0;

		if ( ms->msg_pos )
			needed = GNUTELLA_MSG_SIZE - ms->msg_pos;

		if ( (! ms->msg_pos && ! ms->payload_left &&
		      (bytes_left >= GNUTELLA_MSG_SIZE)) ||
		     (ms->msg_pos && (bytes_left >= needed)) )
			{
			int sz = ms->msg_pos ? needed : GNUTELLA_MSG_SIZE;

			memcpy(&ms->msg[ms->msg_pos],
				&data[ms->current_offset], sz);

			ms->current_offset += sz;
			DissectMessage(ms->msg);
			ms->payload_left = ms->msg_len;
			ms->msg_pos = 0;
			if ( ms->msg_len == 0 )
				SendEvents(ms, (s == orig));
			}

		else if ( (! ms->msg_pos && ! ms->payload_left &&
			   (bytes_left < GNUTELLA_MSG_SIZE)) ||
			  (ms->msg_pos && (bytes_left < needed)) )
			{
			memcpy(&ms->msg[ms->msg_pos], &data[ms->current_offset],
				bytes_left);
			ms->current_offset += bytes_left;
			ms->msg_pos += bytes_left;
			}

		else if ( ms->payload_left )
			{
			unsigned int space =
				ms->payload_len >= GNUTELLA_MAX_PAYLOAD ?
					0 : GNUTELLA_MAX_PAYLOAD - ms->payload_len;
			unsigned int sz =
				(bytes_left < space) ? bytes_left : space;

			if ( space )
				{
				memcpy(&ms->payload[ms->payload_len],
					&data[ms->current_offset], sz);
				ms->payload_len += sz;
				}

			if ( ms->payload_left > bytes_left )
				{
				ms->current_offset += bytes_left;
				ms->payload_left -= bytes_left;
				}
			else
				{
				ms->current_offset += ms->payload_left;
				ms->payload_left = 0;
				SendEvents(ms, (s == orig));
				}
			}
		}
	}


void GnutellaConn::Deliver(TCP_Endpoint* s, int seq, int len, u_char* data)
	{
	if ( http_conn )
		{
		DeliverHTTP(s, seq, len, data);
		return;
		}

	ms = (s == orig) ? orig_msg_state : resp_msg_state;
	ms->current_offset = 0;
	if ( ! Established() )
		{
		DeliverLines(s, seq, len, data);

		if ( http_conn )
			DeliverHTTP(s, seq, len, data);

		else if ( Established() && ms->current_offset < len &&
			  gnutella_binary_msg )
			DeliverMessages(s, seq, len, data);
		}

	else if ( gnutella_binary_msg )
		DeliverMessages(s, seq, len, data);
	}
