// $Id: IOSource.h,v 1.3 2005/01/07 19:45:41 vern Exp $
//
// Interface for classes providing/consuming data during Bro's main loop.

#ifndef iosource_h
#define iosource_h

#include <list>

using namespace std;

class IOSource {
public:
	IOSource()	{ idle = closed = false; }
	virtual ~IOSource()	{}

	// Returns true if source has nothing ready to process.
	bool IsIdle() const	{ return idle; }

	// Returns true if more data is to be expected in the future.
	// Otherwise, source may be removed.
	bool IsOpen() const	{ return ! closed; }

	// Returns select'able fds (leaves args untouched if we don't have
	// selectable fds).
	virtual void GetFds(int* read, int* write, int* except) = 0;


	// The following two methods are only called when either IsIdle()
	// returns false or select() on one of the fds indicates that there's
	// data to process.

	// Returns timestamp associated with next data item.
	virtual double NextTimestamp() = 0;

	// Processes and consumes next data item.
	virtual void Process() = 0;

protected:
	// Derived classed are to set this to true if they have gone dry
	// temporarily.
	bool idle;

	// Derived classed are to set this to true if they have gone dry
	// permanently.
	bool closed;
};

class IOSourceRegistry {
public:
	IOSourceRegistry()	{ call_count = 0; }
	~IOSourceRegistry();

	void Register(IOSource* src);

	// This may block for some time.
	IOSource* FindSoonest(double* ts);

	int Size() const	{ return sources.size(); }

protected:
	// When looking for a source with something to process,
	// every SELECT_FREQUENCY calls we will go ahead and
	// block on a select().
	static const int SELECT_FREQUENCY = 50;

	// Microseconds to wait in an empty select if no source is ready.
	static const int SELECT_TIMEOUT = 100;

	unsigned int call_count;

	struct Source {
		IOSource* src;
		int fd_read;
		int fd_write;
		int fd_except;
	};

	typedef list<Source*> SourceList;
	SourceList sources;
};

extern IOSourceRegistry io_sources;

#endif
