// $Id: NFS.cc,v 1.1 2005/04/21 07:01:54 vern Exp $
//
// Copyright (c) 2005
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include "config.h"

#include "NetVar.h"
#include "XDR.h"
#include "NFS.h"
#include "Event.h"


#define NFS_PROC_NULL 0
#define NFS_PROC_GETATTR 1
#define NFS_PROC_SETATTR 2
#define NFS_PROC_LOOKUP 3
#define NFS_PROC_FSSTAT 18


int NFS_Interp::RPC_BuildCall(RPC_CallInfo* c, const u_char*& buf, int& n)
	{
	if ( c->Program() != 100003 )
		Weird("bad_RPC_program");

	switch ( c->Proc() ) {
	case NFS_PROC_NULL:
		break;

	case NFS_PROC_GETATTR:
		{
		Val* v = ExtractFH(buf, n);
		if ( ! v )
			return 0;
		c->AddVal(v);
		}
		break;

	case NFS_PROC_LOOKUP:
		{
		StringVal* fh = ExtractFH(buf, n);

		int name_len;
		const u_char* name = extract_XDR_opaque(buf, n, name_len);

		if ( ! fh || ! name )
			return 0;

		RecordVal* args = new RecordVal(nfs3_lookup_args);
		args->Assign(0, fh);
		args->Assign(1, new StringVal(new BroString(name, name_len, 0)));
		c->AddVal(args);
		}
		break;

	case NFS_PROC_FSSTAT:
		{
		Val* v = ExtractFH(buf, n);
		if ( ! v )
			return 0;
		c->AddVal(v);
		}
		break;

	default:
		Weird("unknown_NFS_request");
		return 0;
	}

	return 1;
	}

int NFS_Interp::RPC_BuildReply(const RPC_CallInfo* c, int success,
					const u_char*& buf, int& n,
					EventHandlerPtr& event, Val*& reply)
	{
	reply = 0;

	switch ( c->Proc() ) {
	case NFS_PROC_NULL:
		event = success ? nfs_request_null : nfs_attempt_null;
		break;

	case NFS_PROC_GETATTR:
		if ( success )
			{
			uint32 status = extract_XDR_uint32(buf, n);
			if ( ! buf || status != 0 )
				return 0;

			reply = ExtractAttrs(buf, n);
			event = nfs_request_getattr;
			}
		else
			event = nfs_attempt_getattr;

		break;

	case NFS_PROC_LOOKUP:
		if ( success )
			{
			uint32 status = extract_XDR_uint32(buf, n);
			if ( ! buf || status != 0 )
				return 0;

			RecordVal* r = new RecordVal(nfs3_lookup_reply);
			r->Assign(0, ExtractFH(buf, n));
			r->Assign(1, ExtractOptAttrs(buf, n));
			r->Assign(2, ExtractOptAttrs(buf, n));

			reply = r;
			event = nfs_request_lookup;
			}
		else
			{
			reply = ExtractOptAttrs(buf, n);
			event = nfs_attempt_lookup;
			}

		break;

	case NFS_PROC_FSSTAT:
		if ( success )
			{
			uint32 status = extract_XDR_uint32(buf, n);
			if ( ! buf || status != 0 )
				return 0;

			RecordVal* r = new RecordVal(nfs3_fsstat);
			r->Assign(0, ExtractOptAttrs(buf, n));
			r->Assign(1, ExtractLongAsDouble(buf, n)); // tbytes
			r->Assign(2, ExtractLongAsDouble(buf, n)); // fbytes
			r->Assign(3, ExtractLongAsDouble(buf, n)); // abytes
			r->Assign(4, ExtractLongAsDouble(buf, n)); // tfiles
			r->Assign(5, ExtractLongAsDouble(buf, n)); // ffiles
			r->Assign(6, ExtractLongAsDouble(buf, n)); // afiles
			r->Assign(7, ExtractInterval(buf, n)); // invarsec

			reply = r;
			event = nfs_request_fsstat;
			}
		else
			{
			reply = ExtractOptAttrs(buf, n);
			event = nfs_attempt_fsstat;
			}

		break;

	default:
		return 0;
	}

	return 1;
	}

StringVal* NFS_Interp::ExtractFH(const u_char*& buf, int& n)
	{
	int fh_n;
	const u_char* fh = extract_XDR_opaque(buf, n, fh_n, 64);

	if ( ! fh )
		return 0;

	return new StringVal(new BroString(fh, fh_n, 0));
	}

RecordVal* NFS_Interp::ExtractAttrs(const u_char*& buf, int& n)
	{
	RecordVal* attrs = new RecordVal(nfs3_attrs);
	attrs->Assign(0, ExtractCount(buf, n));	// file type
	attrs->Assign(1, ExtractCount(buf, n));	// mode
	attrs->Assign(2, ExtractCount(buf, n));	// nlink
	attrs->Assign(3, ExtractCount(buf, n));	// uid
	attrs->Assign(4, ExtractCount(buf, n));	// gid
	attrs->Assign(5, ExtractLongAsDouble(buf, n));	// size
	attrs->Assign(6, ExtractLongAsDouble(buf, n));	// used
	attrs->Assign(7, ExtractCount(buf, n));	// rdev1
	attrs->Assign(8, ExtractCount(buf, n));	// rdev2
	attrs->Assign(9, ExtractLongAsDouble(buf, n));	// fsid
	attrs->Assign(10, ExtractLongAsDouble(buf, n));	// fileid
	attrs->Assign(11, ExtractTime(buf, n));	// atime
	attrs->Assign(12, ExtractTime(buf, n));	// mtime
	attrs->Assign(13, ExtractTime(buf, n));	// ctime

	return attrs;
	}

RecordVal* NFS_Interp::ExtractOptAttrs(const u_char*& buf, int& n)
	{
	int have_attrs = extract_XDR_uint32(buf, n);

	if ( buf && have_attrs )
		return ExtractAttrs(buf, n);

	else
		return 0;
	}

Val* NFS_Interp::ExtractCount(const u_char*& buf, int& n)
	{
	return new Val(extract_XDR_uint32(buf, n), TYPE_COUNT);
	}

Val* NFS_Interp::ExtractLongAsDouble(const u_char*& buf, int& n)
	{
	return new Val(extract_XDR_uint64_as_double(buf, n), TYPE_DOUBLE);
	}

Val* NFS_Interp::ExtractTime(const u_char*& buf, int& n)
	{
	return new Val(extract_XDR_time(buf, n), TYPE_TIME);
	}

Val* NFS_Interp::ExtractInterval(const u_char*& buf, int& n)
	{
	return new IntervalVal(double(extract_XDR_uint32(buf, n)), 1.0);
	}

void NFS_Interp::Event(EventHandlerPtr f, Val* request, int status, Val* reply)
	{
	if ( ! f )
		{
		Unref(request);
		Unref(reply);
		return;
		}

	val_list* vl = new val_list;

	vl->append(conn->BuildConnVal());
	if ( status == RPC_SUCCESS )
		{
		if ( request )
			vl->append(request);
		if ( reply )
			vl->append(reply);
		}
	else
		{
		vl->append(new Val(status, TYPE_COUNT));
		if ( request )
			vl->append(request);
		}

	conn->ConnectionEvent(f, vl);
	}

NFS_Conn::NFS_Conn(NetSessions* s, HashKey* k, double t,
		const ConnID* id, const struct tcphdr* tp)
: TCP_Connection(s, k, t, id, tp)
	{
	interp = new NFS_Interp(this);
	orig_rpc = resp_rpc = 0;
	}

void NFS_Conn::BuildEndpoints()
	{
	orig_rpc = new TCP_Contents_RPC(interp, orig);
	resp_rpc = new TCP_Contents_RPC(interp, resp);

	orig->AddContentsProcessor(orig_rpc);
	resp->AddContentsProcessor(resp_rpc);
	}

NFS_Conn::~NFS_Conn()
	{
	delete interp;
	}

void NFS_Conn::Done()
	{
	if ( orig_rpc->State() != RPC_COMPLETE &&
	     (orig->state == TCP_CLOSED || orig->prev_state == TCP_CLOSED) &&
	     // Sometimes things like tcpwrappers will immediately close
	     // the connection, without any data having been transferred.
	     // Don't bother flagging these.
	     orig->Size() > 0 )
		Weird("partial_NFS_request");

	interp->Timeout();

	TCP_Connection::Done();
	}
