// $Id: Net.cc,v 1.13 2005/08/23 21:09:17 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2000, 2001, 2002
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include "config.h"

#include <sys/types.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>

#include "NetVar.h"
#include "Sessions.h"
#include "Event.h"
#include "Timer.h"
#include "Var.h"
#include "Logger.h"
#include "Net.h"
#include "TCP_Rewriter.h"
#include "Anon.h"
#include "PacketSort.h"
#include "Serializer.h"

extern "C" {
#include "setsignal.h"
};

extern "C" {
extern int select(int, fd_set *, fd_set *, fd_set *, struct timeval *);
}

PList(PktSrc) pkt_srcs;

// FIXME: We should really merge PktDumper and PacketDumper.
// It's on my to-do [Robin].
PktDumper* pkt_dumper = 0;
PktDumper* pkt_transformed_dumper = 0;

// For trace of rewritten packets
PacketDumper* transformed_pkt_dump = 0;
// For trace of original packets from selected connections
PacketDumper* source_pkt_dump = 0;
int transformed_pkt_dump_MTU = 1514;

int reading_live = 0;
int reading_traces = 0;
int have_pending_timers = 0;
char* user_pcap_filter = 0;
bool using_communication = false;

double network_time = 0.0;	// time according to last packet timestamp
double processing_start_time = 0.0;	// time started working on current pkt
double last_watchdog_proc_time = 0.0;	// value of above during last watchdog
bool terminating = false;	// whether we're done reading and finishing up
#define sort_packets()	(packet_sort_window > 0)

const struct pcap_pkthdr* current_hdr = 0;
const u_char* current_pkt = 0;
int current_dispatched = 0;
PktSrc* current_pktsrc = 0;

RETSIGTYPE watchdog(int /* signo */)
	{
	if ( processing_start_time != 0.0 )
		{
		// The signal arrived while we're processing a packet and/or
		// its corresponding event queue.  Check whether we've been
		// spending too much time, which we take to mean we've wedged.

		// Note that it's subtle how exactly to test this.  In
		// processing_start_time we have the timestamp of the packet
		// we're currently working on.  But that *doesn't* mean that
		// we began work on the packet at that time; we could have
		// begun at a much later time, depending on how long the
		// packet filter waited (to fill its buffer) before handing
		// up this packet.  So what we require is that the current
		// processing_start_time matches the processing_start_time we
		// observed last time the watchdog went off.  If so, then
		// we've been working on the current packet for at least
		// watchdog_interval seconds.

		if ( processing_start_time == last_watchdog_proc_time )
			{
			// snprintf() calls alloc/free routines if you use %f!
			// We need to avoid doing that given we're in a single
			// handler and the allocation routines are not
			// reentrant.

			double ct = current_time();

			int int_ct = int(ct);
			int frac_ct = int((ct - int_ct) * 1e6);

			int int_pst = int(processing_start_time);
			int frac_pst =
				int((processing_start_time - int_pst) * 1e6);

			char msg[512];
			snprintf(msg, sizeof(msg),
				"**watchdog timer expired, t = %d.%06d, start = %d.%06d, dispatched = %d",
				int_ct, frac_ct, int_pst, frac_pst,
				current_dispatched);

			bro_logger->Log(msg);
			run_time("watchdog timer expired");

			if ( current_hdr && pkt_dumper)
				pkt_dumper->Dump(current_hdr, current_pkt);
			net_get_final_stats();
			net_finish(0);

			abort();
			exit(1);
			}
		}

	last_watchdog_proc_time = processing_start_time;

	(void) alarm(watchdog_interval);
	return RETSIGVAL;
	}

void net_init(name_list& interfaces, name_list& readfiles,
	        const char* writefile, const char* transformed_writefile,
	        const char* filter, const char* secondary_filter,
		int do_watchdog)
	{
	init_net_var();

	if ( readfiles.length() > 0 )
		{
		reading_live = 0;
		reading_traces = 1;

		for ( int i = 0; i < readfiles.length(); ++i )
			{
			PktFileSrc* ps = new PktFileSrc(readfiles[i], filter);

			if ( ! ps->IsOpen() )
				{
				fprintf(stderr, "%s: problem with trace file %s - %s\n",
					prog, readfiles[i], ps->ErrorMsg());
				exit(1);
				}
			else
				{
				pkt_srcs.append(ps);
				io_sources.Register(ps);
				}

			if ( secondary_filter )
				{
				// We use a second PktFileSrc for the
				// secondary path.
				PktFileSrc* ps = new PktFileSrc(readfiles[i],
							secondary_filter,
							TYPE_FILTER_SECONDARY);

				if ( ! ps->IsOpen() )
					{
					fprintf(stderr, "%s: problem with trace file %s - %s\n",
						prog, readfiles[i],
						ps->ErrorMsg());
					exit(1);
					}
				else
					{
					pkt_srcs.append(ps);
					io_sources.Register(ps);
					}

				ps->AddSecondaryTablePrograms();
				}
			}
		}

	else if ( interfaces.length() > 0 )
		{
		reading_live = 1;
		reading_traces = 0;

		for ( int i = 0; i < interfaces.length(); ++i )
			{
			PktInterfaceSrc* ps =
				new PktInterfaceSrc(interfaces[i], filter);

			if ( ! ps->IsOpen() )
				{
				fprintf(stderr, "%s: problem with interface %s - %s\n",
					prog, interfaces[i], ps->ErrorMsg());
				exit(1);
				}
			else
				{
				pkt_srcs.append(ps);
				io_sources.Register(ps);
				}

			if ( secondary_filter )
				{
				PktInterfaceSrc* ps =
					new PktInterfaceSrc(interfaces[i],
						filter, TYPE_FILTER_SECONDARY);

				if ( ! ps->IsOpen() )
					{
					fprintf(stderr, "%s: problem with interface %s - %s\n",
						prog, interfaces[i],
						ps->ErrorMsg());
					exit(1);
					}
				else
					{
					pkt_srcs.append(ps);
					io_sources.Register(ps);
					}

				ps->AddSecondaryTablePrograms();
				}
			}
		}

	else
		// have_pending_timers = 1, possibly.  We don't set
		// that here, though, because at this point we don't know
		// whether the user's bro_init() event will indeed set
		// a timer.
		reading_traces = reading_live = 0;

	if ( writefile )
		{
		// ### This will fail horribly if there are multiple
		// interfaces with different-lengthed media.
		pkt_dumper = new PktDumper(writefile);
		if ( pkt_dumper->IsError() )
			{
			fprintf(stderr, "%s: can't open write file \"%s\" - %s\n",
				prog, writefile, pkt_dumper->ErrorMsg());
			exit(1);
			}
		}

	if ( transformed_writefile )
		{
		pkt_transformed_dumper = new PktDumper(transformed_writefile);
		if ( pkt_transformed_dumper->IsError() )
			{
			fprintf(stderr, "%s: can't open trace transformation write file \"%s\" - %s\n",
				prog, writefile,
				pkt_transformed_dumper->ErrorMsg());
			exit(1);
			}

		transformed_pkt_dump =
			new PacketDumper(pkt_transformed_dumper->PcapDumper());

		// If both -A and -w are specified, -A will be the transformed
		// trace file and -w will be the source packet trace file.
		// Otherwise the packets will go to the same file.
		if ( pkt_dumper )
			source_pkt_dump =
				new PacketDumper(pkt_dumper->PcapDumper());
		}

	else if ( pkt_dumper )
		transformed_pkt_dump =
			new PacketDumper(pkt_dumper->PcapDumper());

	if ( anonymize_ip_addr )
		init_ip_addr_anonymizers();
	else
		for ( int i = 0; i < NUM_ADDR_ANONYMIZATION_METHODS; ++i )
			ip_anonymizer[i] = 0;

	packet_sorter = new PacketSortGlobalPQ();

	sessions = new NetSessions();

	if ( do_watchdog )
		{
		// Set up the watchdog to make sure we don't wedge.
		(void) setsignal(SIGALRM, watchdog);
		(void) alarm(watchdog_interval);
		}
	}

void net_run()
	{
	while ( io_sources.Size() || have_pending_timers )
		{
		double ts;
		IOSource* src =
			io_sources.Size() ? io_sources.FindSoonest(&ts) : 0;

		if ( src )
			{
			if ( ts > network_time )
				network_time = ts;

			processing_start_time = network_time;

			{
			SegmentProfiler(segment_logger, "expiring-timers");
			current_dispatched = timer_mgr->Advance(network_time, max_timer_expires);
			}

			src->Process();
			}

		else if ( reading_live )
			{
			// Take advantage of the lull to get up to
			// date on timers and events.
			timer_mgr->Advance(network_time, max_timer_expires);
			}

		else if ( have_pending_timers )
			{
			// Take advantage of the lull to get up to
			// date on timers and events.  Because we only
			// have timers as sources, going to sleep here
			// doesn't risk blocking on other inputs.
			network_time = current_time();
			timer_mgr->Advance(network_time, max_timer_expires);

			// Avoid busy-waiting - pause for 100 ms.
			// We pick a sleep value of 100 msec that buys
			// us a lot of idle time, but doesn't delay near-term
			// timers too much.  (Delaying them somewhat is okay,
			// since Bro timers are not high-precision anyway.)
			usleep(100000);
			}

		mgr.Drain();

		processing_start_time = 0.0;	// = "we're not processing now"
		current_dispatched = 0;

		// Should we put the signal handling into an IOSource?
		extern void termination_signal();
		if ( signal_val == SIGTERM || signal_val == SIGINT )
			// We received a signal while processing the
			// current packet and its related events.
			termination_signal();

		current_hdr = 0;	// done with these
		current_pkt = 0;
		current_pktsrc = 0;

		if ( ! reading_traces )
			// Check whether we have timers scheduled for
			// the future on which we need to wait.
			have_pending_timers = timer_mgr->Size() > 0;
		}

	// Get the final statistics now, and not when net_finish() is
	// called, since that might happen quite a bit in the future
	// due to expiring pending timers, and we don't want to ding
	// for any packets dropped beyond this point.
	net_get_final_stats();
	}

void net_get_final_stats()
	{
	loop_over_list(pkt_srcs, i)
		{
		PktSrc* ps = pkt_srcs[i];

		if ( ps->IsLive() )
			{
			struct PktSrc::Stats s;
			ps->Statistics(&s);
			fprintf(stderr, "%d packets received on interface %s, %d dropped\n",
					s.received, ps->Interface(), s.dropped);
			}
		}
	}

void net_finish(int drain_events)
	{
	if ( drain_events )
		{
		if ( sessions )
			sessions->Drain();

		mgr.Drain();

		if ( sessions )
			sessions->Done();
		}

	delete pkt_dumper;
	delete pkt_transformed_dumper;

#ifdef USE_UHASH
	DEBUG_MSG("uhash: %d/%d\n", hash_cnt_uhash, hash_cnt_all);
#endif
#ifdef DEBUG_BRO
	extern int num_packets_held, num_packets_cleaned;
	DEBUG_MSG("packets clean up: %d/%d\n", num_packets_cleaned, num_packets_held);
#endif
	}

void net_delete()
	{
	delete sessions;
	delete packet_sorter;

	// Can't put this in net_finish() because packets might be
	// dumped when connections are deleted.
	if ( transformed_pkt_dump )
		delete transformed_pkt_dump;

	for ( int i = 0; i < NUM_ADDR_ANONYMIZATION_METHODS; ++i )
		delete ip_anonymizer[i];
	}

// net_packet_match
//
// Description:
//  - Checks if a packet matches a filter. It just wraps up a call to
//    [pcap.h's] bpf_filter().
//
// Inputs:
//  - fp: a BPF-compiled filter
//  - pkt: a pointer to the packet
//  - len: the original packet length
//  - caplen: the captured packet length. This is pkt length
//
// Output:
//  - return: 1 if the packet matches the filter, 0 otherwise

int net_packet_match(BPF_Program* fp, const u_char* pkt,
		     u_int len, u_int caplen)
	{
	// NOTE: I don't like too much un-const'ing the pkt variable.
	return bpf_filter(fp->GetProgram()->bf_insns, (u_char*) pkt, len, caplen);
	}
