// Copyright (c) 1998, 1999, 2001, 2002
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef reassem_h
#define reassem_h

#include "Active.h"
#include "Obj.h"

class DataBlock {
public:
	DataBlock(const u_char* data, int size, int seq,
			DataBlock* next, DataBlock* prev);

	~DataBlock();

	int Size() const	{ return upper - seq; }

	DataBlock* next;	// next block with higher seq #
	DataBlock* prev;	// previous block with lower seq #
	int seq, upper;
	u_char* block;
};


enum ReassemblerType { REASSEM_IP, REASSEM_TCP };

class Reassembler : public BroObj {
public:
	Reassembler(int init_seq, const uint32* ip_addr,
			ReassemblerType arg_type);
	virtual ~Reassembler();

	void NewBlock(double t, int seq, int len, const u_char* data);

	// Throws away all blocks up to seq.  Returns 1 if they were
	// all in-sequence, 0 if not.
	int TrimToSeq(int seq);

	// Delete all held blocks.
	void ClearBlocks();

	int HasBlocks() const		{ return blocks != 0; }
	int LastReassemSeq() const	{ return last_reassem_seq; }

	int TotalSize() const;	// number of bytes buffered up

	void Describe(ODesc* d) const;

	bool Serialize(SerialInfo* info) const;
	static Reassembler* Unserialize(UnserialInfo* info);

	// Sum over all data buffered in some reassembler.
	static unsigned int TotalMemoryAllocation()	{ return total_size; }

protected:
	Reassembler()	{ }

	DECLARE_ABSTRACT_SERIAL(Reassembler);

	friend class DataBlock;

	virtual void Undelivered(int up_to_seq);

	virtual void BlockInserted(DataBlock* b) = 0;
	virtual void Overlap(const u_char* b1, const u_char* b2, int n) = 0;

	DataBlock* AddAndCheck(DataBlock* b, int seq,
				int upper, const u_char* data);

	DataBlock* blocks;
	DataBlock* last_block;
	int last_reassem_seq;
	int trim_seq;	// how far we've trimmed

	ReassemblyPolicy policy;

	static unsigned int total_size;
};

inline DataBlock::~DataBlock()
	{
	Reassembler::total_size -= pad_size(upper - seq) + padded_sizeof(DataBlock);
	delete [] block;
	}

#endif
