// $Id: RemoteSerializer.cc,v 1.17 2005/09/02 23:00:06 vern Exp $
//
// Processes involved in the communication:
//
//	 (Local-Parent) <-> (Local-Child) <-> (Remote-Child) <-> (Remote-Parent)
//
// Message types (for parent<->child communication the CMsg's peer indicates
// about whom we're talking).
//
// Communication protocol version
//	VERSION <version> <cache_size> <data-format-version> <reserved>
//
// Send serialization
//	SERIAL <serialization>
//
// Terminate(d) connection
//	CLOSE
//
// Close(d) all connections
//	CLOSE_ALL
//
// Connect to remote side
//	CONNECT_TO <id-of-new-peer> <ip> <port> <retry-interval> <use-ssl>
//
// Connected to remote side
//	CONNECTED <ip> <port>
//
// Request events from remote side
//	REQUEST_EVENTS <list of events>
//
// Request synchronization of IDs with remote side
//	REQUEST_SYNC
//
// Listen for connection on ip/port (ip may be INADDR_ANY)
//	LISTEN <ip> <port> <use_ssl>
//
// Close listen ports.
//	LISTEN_STOP
//
// Error caused by host
//	ERROR <msg>
//
// Some statistics about the given peer connection
//	STATS <string>
//
// Requests to set a new capture_filter
//	CAPTURE_FILTER <string>
//
// Valid messages between processes:
//
//	Main -> Child
//		CONNECT_TO
//		REQUEST
//		SERIAL
//		CLOSE
//		CLOSE_ALL
//		LISTEN
//		LISTEN_STOP
//		CAPTURE_FILTER
//		VERSION
// 		REQUEST_SYNC
// 		HANDSHAKE_COMPLETE
//
//	Child -> Main
//		CONNECTED
//		REQUEST
//		SERIAL
//		CLOSE
//		ERROR
//		STATS
//		VERSION
//		CAPTURE_FILTER
// 		REQUEST_SYNC
// 		HANDSHAKE_COMPLETE
//
//	Child <-> Child
//		VERSION
//		SERIAL
//		REQUEST
//		CAPTURE_FILTER
// 		REQUEST_SYNC
// 		HANDSHAKE_COMPLETE
//
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <signal.h>
#include <strings.h>
#include <stdarg.h>

#include "config.h"
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <sys/resource.h>

#include "RemoteSerializer.h"
#include "Func.h"
#include "EventRegistry.h"
#include "Event.h"
#include "Net.h"
#include "NetVar.h"
#include "Scope.h"
#include "Sessions.h"

extern "C" {
#include "setsignal.h"
};

// Gets incremented each time there's an incompatible change
// to the communication internval.
static const unsigned short PROTOCOL_VERSION = 0x04;

static const char MSG_NONE = 0x00;
static const char MSG_VERSION = 0x01;
static const char MSG_SERIAL = 0x02;
static const char MSG_CLOSE = 0x03;
static const char MSG_CLOSE_ALL = 0x04;
static const char MSG_ERROR = 0x05;
static const char MSG_CONNECT_TO = 0x06;
static const char MSG_CONNECTED = 0x07;
static const char MSG_REQUEST_EVENTS = 0x08;
static const char MSG_LISTEN = 0x09;
static const char MSG_LISTEN_STOP = 0x0a;
static const char MSG_STATS = 0x0b;
static const char MSG_CAPTURE_FILTER = 0x0c;
static const char MSG_REQUEST_SYNC = 0x0d;
static const char MSG_HANDSHAKE_COMPLETE = 0x0e;

#ifdef DEBUG
# define DEBUG_COMM(msg) DBG_LOG(DBG_COMM, msg)
#else
# define DEBUG_COMM(msg)
#endif

#define READ_CHUNK(io, c, do_if_eof) \
	{ \
	if ( ! io->Read(&c) ) \
		{ \
		if ( io->Eof() ) \
			{ \
			do_if_eof; \
			} \
		else \
			Error(fmt("can't read data chunk: %s", io->Error())); \
		return false; \
		} \
	\
	if ( ! c ) \
		return true; \
	}

#define READ_CHUNK_FROM_CHILD(c) \
	{ \
	if ( ! io->Read(&c) ) \
		{ \
		if ( io->Eof() ) \
			ChildDied(); \
		else \
			Error(fmt("can't read data chunk: %s", io->Error())); \
		return false; \
		} \
	\
	if ( ! c ) \
		{ \
		idle = true; \
		return true; \
		} \
	idle = false; \
	}


#ifdef DEBUG
const char* msgToStr(int msg)
	{
# define MSG_STR(x) case x: return #x;
	switch ( msg ) {
	MSG_STR(MSG_VERSION)
	MSG_STR(MSG_NONE)
	MSG_STR(MSG_SERIAL)
	MSG_STR(MSG_CLOSE)
	MSG_STR(MSG_CLOSE_ALL)
	MSG_STR(MSG_ERROR)
	MSG_STR(MSG_CONNECT_TO)
	MSG_STR(MSG_CONNECTED)
	MSG_STR(MSG_REQUEST_EVENTS)
	MSG_STR(MSG_LISTEN)
	MSG_STR(MSG_LISTEN_STOP)
	MSG_STR(MSG_STATS)
	MSG_STR(MSG_CAPTURE_FILTER)
	MSG_STR(MSG_REQUEST_SYNC)
	MSG_STR(MSG_HANDSHAKE_COMPLETE)
	default:
		return "UNKNOWN_MSG";
	}
	}
#endif

// Start of every message between two processes.
typedef struct {
	char type;
	RemoteSerializer::PeerID peer; // arbitrary for remote communication.
} CMsg;

static bool sendCMsg(ChunkedIO* io, char msg_type, RemoteSerializer::PeerID id)
	{
	// We use the new[] operator here to avoid mismatches
	// when deleting the data.
	CMsg* msg = (CMsg*) new char[sizeof(CMsg)];

	msg->type = msg_type;
	msg->peer = htonl(id);

	ChunkedIO::Chunk* c = new ChunkedIO::Chunk;
	c->len = sizeof(CMsg);
	c->data = (char*) msg;

	return io->Write(c);
	}

static bool sendToIO(ChunkedIO* io, ChunkedIO::Chunk* c)
	{
	if ( ! io->Write(c) )
		{
		warn(fmt("can't send chunk: %s", io->Error()));
		return false;
		}

	return true;
	}

static bool sendToIO(ChunkedIO* io, char msg_type, RemoteSerializer::PeerID id, const char* str, int len = -1)
	{
	if ( ! sendCMsg(io, msg_type, id) )
		{
		warn(fmt("can't send message of type %d: %s", msg_type, io->Error()));
		return false;
		}

	ChunkedIO::Chunk* c = new ChunkedIO::Chunk;
	c->len = len >= 0 ? len : strlen(str) + 1;
	c->data = const_cast<char *>(str);
	return sendToIO(io, c);
	}

static bool sendToIO(ChunkedIO* io, char msg_type, RemoteSerializer::PeerID id, int nargs, va_list ap)
	{
	if ( ! sendCMsg(io, msg_type, id) )
		{
		warn(fmt("can't send message of type %d: %s", msg_type, io->Error()));
		return false;
		}

	if ( nargs == 0 )
		return true;

	uint32* args = new uint32[nargs];

	for ( int i = 0; i < nargs; i++ )
		args[i] = htonl(va_arg(ap, uint32));

	ChunkedIO::Chunk* c = new ChunkedIO::Chunk;
	c->len = sizeof(uint32) * nargs;
	c->data = (char*) args;

	return sendToIO(io, c);
	}

#ifdef DEBUG
static inline char* fmt_uint32s(int nargs, va_list ap)
	{
	static char buf[512];
	char* p = buf;
	*p = '\0';
	for ( int i = 0; i < nargs; i++ )
		p += snprintf(p, sizeof(buf) - (p - buf),
				" 0x%08x", va_arg(ap, uint32));
	buf[511] = '\0';
	return buf;
	}
#endif


static inline const char* ip2a(uint32 ip)
	{
	static char buffer[32];
	struct in_addr addr;

	addr.s_addr = htonl(ip);

	return inet_ntop(AF_INET, &addr, buffer, 32);
	}

static pid_t child_pid = 0;

bool RemoteSerializer::IsConnectedPeer(PeerID id)
	{
	if ( id == PEER_NONE )
		return true;

	return LookupPeer(id, true) != 0;
	}

RemoteSerializer::RemoteSerializer()
	{
	initialized = false;
	current_peer = 0;
	state = MSG_NONE;
	id_counter = 1;
	listening = false;
	ignore_accesses = false;
	events_received = 0;
	events_sent = 0;
	error_fd = -1;
	io = 0;
	closed = false;
	}

RemoteSerializer::~RemoteSerializer()
	{
	if ( child_pid )
		{
		kill(child_pid, SIGTERM);
		waitpid(child_pid, 0, 0);
		}

	delete io;
	}

void RemoteSerializer::Init()
	{
	if ( initialized || reading_traces )
		return;

	error_fd = open("comm.log", O_WRONLY | O_CREAT | O_TRUNC, 0600);
	if ( error_fd < 0 )
		{
		error("can't open comm.log: %s", strerror(errno));
		exit(1);
		}

	Fork();

	io_sources.Register(this);

	Log(fmt("communication started, parent pid is %d, child pid is %d", getpid(), child_pid));
	initialized = 1;
	}

void RemoteSerializer::Fork()
	{
	if ( child_pid )
		return;

	// If we are re-forking, remove old entries
	loop_over_list(peers, i)
		RemovePeer(peers[i]);

	// Create pipe for communication between parent and child.
	int pipe[2];

	if ( socketpair(AF_UNIX, SOCK_STREAM, 0, pipe) < 0 )
		{
		Error(fmt("can't create pipe: %s", strerror(errno)));
		return;
		}

	child_pid = 0;

	int pid = fork();

	if ( pid < 0 )
		{
		Error(fmt("can't fork: %s", strerror(errno)));
		return;
		}

	if ( pid > 0 )
		{
		// Parent
		child_pid = pid;

		io = new ChunkedIOFd(pipe[0], "parent->child");
		if ( ! io->Init() )
			{
			Error(fmt("can't init child io: %s", io->Error()));
			exit(1); // FIXME: Better way to handle this?
			}

		close(pipe[1]);

		return;
		}
	else
		{ // child
		SocketComm child(error_fd);

		ChunkedIOFd* io = new ChunkedIOFd(pipe[1], "child->parent");
		if ( ! io->Init() )
			{
			Error(fmt("can't init parent io: %s", io->Error()));
			exit(1);
			}

		child.SetParentIO(io);
		close(pipe[0]);

		// Close file descriptors.
		close(0);
		close(1);
		close(2);

		// Be nice.
		setpriority(PRIO_PROCESS, 0, 5);

		child.Run();
		internal_error("cannot be reached");
		}
	}

RemoteSerializer::PeerID RemoteSerializer::Connect(addr_type ip, uint16 port,
						double retry, bool use_ssl)
	{
	if ( reading_traces )
		return true;

	if ( ! initialized )
		internal_error("remote serializer not initialized");

#ifdef BROv6
	if ( ! is_v4_addr(ip) )
		Error("inter-Bro communication not supported over IPv6");

	uint32 ip4 = to_v4_addr(ip);
#else
	uint32 ip4 = ip;
#endif

	ip4 = ntohl(ip4);

	if ( ! child_pid )
		Fork();

	Peer* p = AddPeer(ip4, port);
	p->orig = true;

	if ( ! SendToChild(MSG_CONNECT_TO, p, 5, p->id,
				ip4, port, uint32(retry), use_ssl) )
		{
		RemovePeer(p);
		return false;
		}

	p->state = Peer::PENDING;
	return p->id;
	}

bool RemoteSerializer::CloseConnection(Peer* peer)
	{
	return SendToChild(MSG_CLOSE, peer, 0);
	}

bool RemoteSerializer::RequestSync(PeerID id)
	{
	if ( reading_traces )
		return true;

	Peer* peer = LookupPeer(id, true);
	return SendToChild(MSG_REQUEST_SYNC, peer, 0);
	}

bool RemoteSerializer::RequestEvents(PeerID id, RE_Matcher* pattern)
	{
	if ( reading_traces )
		return true;

	Peer* peer = LookupPeer(id, true);

	EventRegistry::string_list* handlers = event_registry->Match(pattern);

	// Concat the handlers' names.
	int len = 0;
	loop_over_list(*handlers, i)
		len += strlen((*handlers)[i]) + 1;

	if ( ! len )
		{
		Log("warning: No events to request");
		return true;
		}

	char* data = new char[len];
	char* d = data;
	loop_over_list(*handlers, j)
		{
		for ( const char* p = (*handlers)[j]; *p; *d++ = *p++ )
			;
		*d++ = '\0';
		}

	delete handlers;

	return SendToChild(MSG_REQUEST_EVENTS, peer, data, len);
	}

bool RemoteSerializer::SetAcceptState(PeerID id, bool accept)
	{
	Peer* p = LookupPeer(id, false);
	if ( ! p )
		return true;

	p->accept_state = accept;
	return true;
	}

bool RemoteSerializer::CompleteHandshake(PeerID id)
	{
	Peer* p = LookupPeer(id, false);
	if ( ! p )
		return true;

	if ( p->handshake == Peer::NONE )
		p->handshake = Peer::SENT;
	else if ( p->handshake == Peer::RECEIVED )
		p->handshake = Peer::COMPLETED;

	Log(fmt("completing handshake (state is %d)", p->handshake),
	    current_peer);

	return SendToChild(MSG_HANDSHAKE_COMPLETE, p, 0);
	}

bool RemoteSerializer::SendCall(SerialInfo* info, PeerID id,
					const char* name, val_list* vl)
	{
	if ( reading_traces )
		return true;

	Peer* peer = LookupPeer(id, true);
	if ( ! peer )
		return false;

	if ( peer->handshake != Peer::COMPLETED )
		return false;

	if ( ! SendToChild(MSG_SERIAL, peer, 0) )
		return false;

	++events_sent;
	SetCache(peer->cache_out);
	return Serialize(info, name, vl);
	}

bool RemoteSerializer::SendAccess(SerialInfo* info, PeerID id, const StateAccess& access)
	{
	Peer* peer = LookupPeer(id, true);
	if ( ! peer )
		return false;

	if ( ! peer->sync )
		return true;

#ifdef DEBUG
	ODesc desc;
	access.Describe(&desc);
	DBG_LOG(DBG_COMM, "Sending %s", desc.Description());
#endif

	if ( ! SendToChild(MSG_SERIAL, peer, 0) )
		return false;

	SetCache(peer->cache_out);
	return Serialize(info, access);
	}

bool RemoteSerializer::SendAccess(SerialInfo* info, const StateAccess& access)
	{
	if ( ! IsOpen() || ignore_accesses )
		return true;

	// A real broadcast would be nice here. But the different peers have
	// different serialization caches, so we cannot simply send the same
	// serialization to all of them ...
	loop_over_list(peers, i)
		{
		if ( peers[i]->handshake != Peer::COMPLETED )
			continue;

		SerialInfo new_info(*info);
		if ( ! SendAccess(&new_info, peers[i]->id, access) )
			return false;
		}

	return true;
	}

bool RemoteSerializer::SendAllSynchronized(SerialInfo* info, Peer* peer)
	{
	loop_over_list(sync_ids, i)
		{
		Val* mv = sync_ids[i]->ID_Val();
		assert(mv);

		// Send an assigment to the user-visible ID.
		StateAccess access(OP_ASSIGN, sync_ids[i], mv);
		if ( ! SendAccess(info, peer->id, access) )
				return false;
		}

	return true;
	}

bool RemoteSerializer::SendID(SerialInfo* info, PeerID pid, const ID& id)
	{
	if ( reading_traces )
		return true;

	Peer* peer = LookupPeer(pid, true);
	if ( ! peer )
		return false;

	if ( ! SendToChild(MSG_SERIAL, peer, 0) )
		return false;

	SetCache(peer->cache_out);
	return Serialize(info, id);
	}

bool RemoteSerializer::SendConnection(SerialInfo* info, PeerID id, const Connection& c)
	{
	if ( reading_traces )
		return true;

	Peer* peer = LookupPeer(id, true);
	if ( ! peer )
		return false;

	if ( ! SendToChild(MSG_SERIAL, peer, 0) )
		return false;

	SetCache(peer->cache_out);
	return Serialize(info, c);
	}

bool RemoteSerializer::SendCaptureFilter(PeerID id, const char* filter)
	{
	if ( reading_traces )
		return true;

	Peer* peer = LookupPeer(id, true);
	if ( ! peer )
		return false;

	return SendToChild(MSG_CAPTURE_FILTER, peer, copy_string(filter));
	}

bool RemoteSerializer::SendPacket(SerialInfo* info, PeerID id, const Packet& p)
	{
	if ( ! reading_live )
		return true;

	Peer* peer = LookupPeer(id, true);
	if ( ! peer )
		return false;

	if ( ! SendToChild(MSG_SERIAL, peer, 0) )
		return false;

	SetCache(peer->cache_out);
	return Serialize(info, p);
	}

bool RemoteSerializer::Listen(addr_type ip, uint16 port, bool expect_ssl)
	{
	if ( reading_traces )
		return true;

#ifndef USE_OPENSSL
	if ( expect_ssl )
		{
		Error("listening for SSL connections requested, but SSL support is not compiled in");
		return false;
		}
#endif

	if ( ! initialized )
		internal_error("remote serializer not initialized");

#ifdef BROv6
	if ( ! is_v4_addr(ip) )
		Error("inter-Bro communication not supported over IPv6");

	uint32 ip4 = to_v4_addr(ip);
#else
	uint32 ip4 = ip;
#endif

	ip4 = ntohl(ip4);

	if ( ! SendToChild(MSG_LISTEN, 0, 3, ip4, port, expect_ssl) )
		return false;

	listening = true;
	closed = false;
	return true;
	}

bool RemoteSerializer::StopListening()
	{
	if ( ! listening )
		return true;

	if ( ! SendToChild(MSG_LISTEN_STOP, 0, 0) )
		return false;

	listening = false;
	closed = ! IsActive();
	return true;
	}

void RemoteSerializer::Register(ID* id)
	{
	DBG_LOG(DBG_STATE, "&synchronized %s", id->Name());
	Unregister(id);
	Ref(id);
	sync_ids.append(id);
	}

void RemoteSerializer::Unregister(ID* id)
	{
	loop_over_list(sync_ids, i)
		if ( streq(sync_ids[i]->Name(), id->Name()) )
			{
			Unref(sync_ids[i]);
			sync_ids.remove_nth(i);
			break;
			}
	}

void RemoteSerializer::GetFds(int* read, int* write, int* except)
	{
	*read = io->Fd();

	if ( io->CanWrite() )
		*write = io->Fd();
	}

double RemoteSerializer::NextTimestamp()
	{
	Poll(false);
	return events.length() ? events[0]->time : -1;
	}

void RemoteSerializer::Process()
	{
	Poll(false);
	if ( events.length() )
		{
		BufferedEvent* be = events[0];
		::Event* event = new ::Event(be->handler, be->args, be->src);

		mgr.Dispatch(event, true); // don't forward received events.
		delete be;
		events.remove_nth(0);
		}
	}

bool RemoteSerializer::Poll(bool may_block)
	{
	if ( ! child_pid )
		return true;

	io->Flush();
	idle = false;

	switch ( state ) {
	case MSG_NONE:
		{
		// CMsg follows
		ChunkedIO::Chunk* c;
		READ_CHUNK_FROM_CHILD(c);

		CMsg* msg = (CMsg*) c->data;
		current_peer = LookupPeer(ntohl(msg->peer), false);
		current_id = ntohl(msg->peer);

		DEBUG_COMM(fmt("parent: %s from child; peer is #%d", msgToStr(msg->type), ntohl(msg->peer)));

		switch ( msg->type ) {
		case MSG_CLOSE:
			if ( current_peer )
				PeerDisconnected(current_peer);
			break;

		case MSG_HANDSHAKE_COMPLETE:
			return ProcessHandshakeComplete();

		default:
			// Msg with more arguments.

			// As long as we haven't finished the version
			// handshake, no other messages than MSG_VERSION
			// are allowed from peer.
			if ( current_peer && ! current_peer->got_version &&
			     msg->type != MSG_CONNECTED &&
			     msg->type != MSG_ERROR &&
			     msg->type != MSG_VERSION &&
			     msg->type != MSG_STATS )
				{
				Log("peer did not send version", current_peer);
				CloseConnection(current_peer);
				}

			state = msg->type;
		}

		delete [] c->data;
		delete c;

		return true;
		}

	case MSG_CONNECTED:
		return ProcessConnected();

	case MSG_SERIAL:
		// Serialization follows
		return ProcessSerialization();

	case MSG_REQUEST_EVENTS:
		// List of events follows
		return ProcessRequestEventsMsg();

	case MSG_REQUEST_SYNC:
		return ProcessRequestSyncMsg();

	case MSG_ERROR:
		return ProcessErrorMsg();

	case MSG_STATS:
		return ProcessStatsMsg();

	case MSG_CAPTURE_FILTER:
		return ProcessCaptureFilterMsg();

	case MSG_VERSION:
		return ProcessVersionMsg();

	default:
		internal_error(fmt("unexpected msg type: %d", (int) state));
	}
	}

void RemoteSerializer::PeerDisconnected(Peer* peer)
	{
	if ( peer->state == Peer::CLOSED || peer->state == Peer::INIT )
		return;

	if ( peer->state == Peer::PENDING )
		{
		peer->state = Peer::CLOSED;
		Log("could not connect", peer);
		return;
		}

	Log("peer disconnected", peer);
	peer->state = Peer::CLOSED;
	UnregisterHandlers(peer);
	RaiseEvent(remote_connection_closed, peer);
	}

void RemoteSerializer::PeerConnected(Peer* peer)
	{
	if ( peer->state == Peer::CONNECTED )
		return;

	peer->state = Peer::CONNECTED;
	peer->handshake = Peer::NONE;
	peer->cache_in->Clear();
	peer->cache_out->Clear();

	if ( ! SendToChild(MSG_VERSION, peer, 4, PROTOCOL_VERSION,
			   peer->cache_out->GetMaxCacheSize(),
			   DATA_FORMAT_VERSION, 0) )
		{
		Log("can't send version message");
		CloseConnection(peer);
		return;
		}

	Log("peer connected", peer);
	RaiseEvent(remote_connection_established, current_peer);
	}

RecordVal* RemoteSerializer::MakePeerVal(Peer* peer)
	{
	RecordVal* v = new RecordVal(::peer);
	v->Assign(0, new Val(uint32(peer->id), TYPE_COUNT));
	// Sic! Network order for AddrVal, host order for PortVal.
	v->Assign(1, new AddrVal(htonl(peer->ip)));
	v->Assign(2, new PortVal(peer->port));
	v->Assign(3, new Val(false, TYPE_BOOL));
	v->Assign(4, new StringVal(""));	// set when received
	return v;
	}

RemoteSerializer::Peer* RemoteSerializer::AddPeer(uint32 ip, uint16 port, PeerID id)
	{
	Peer* peer = new Peer;
	peer->id = id != PEER_NONE ? id : id_counter++;
	peer->ip = ip;
	peer->port = port;
	peer->state = Peer::INIT;
	peer->orig = false;
	peer->sync = false;
	peer->got_version = false;
	peer->accept_state = false;
	peer->handshake = Peer::NONE;
	peer->val = MakePeerVal(peer);
	peer->cache_in = new SerializationCache(MAX_CACHE_SIZE);
	peer->cache_out = new SerializationCache(MAX_CACHE_SIZE);

	peers.append(peer);
	Log("added peer", peer);

	return peer;
	}

void RemoteSerializer::UnregisterHandlers(Peer *peer)
	{
	// Unregister the peers for the EventHandlers.
	loop_over_list(peer->handlers, i)
		{
		peer->handlers[i]->RemoveRemoteHandler(peer->id);
		}
	}

void RemoteSerializer::RemovePeer(Peer* peer)
	{
	peers.remove(peer);
	UnregisterHandlers(peer);

	Log("removed peer", peer);

	int id = peer->id;
	Unref(peer->val);
	delete peer->cache_in;
	delete peer->cache_out;
	delete peer;

	closed = ! IsActive();
	}

RemoteSerializer::Peer* RemoteSerializer::LookupPeer(PeerID id, bool only_if_connected)
	{
	Peer* peer = 0;
	loop_over_list(peers, i)
		if ( peers[i]->id == id )
			{
			peer = peers[i];
			break;
			}

	if ( ! only_if_connected || peer->state == Peer::CONNECTED )
		return peer;
	else
		return 0;
	}

bool RemoteSerializer::ProcessVersionMsg()
	{
	ChunkedIO::Chunk* c;
	READ_CHUNK_FROM_CHILD(c);
	state = MSG_NONE;

	uint32* args = (uint32*) c->data;
	uint32 version = ntohl(args[0]);
	uint32 data_version = ntohl(args[2]);

	if ( PROTOCOL_VERSION != version )
		{
		Log(fmt("remote protocol version mismatch: got %d, but expected %d\n",
				version, PROTOCOL_VERSION), current_peer);
		CloseConnection(current_peer);
		return true;
		}

	// For backwards compatibility, data_version may be null.
	if ( data_version && DATA_FORMAT_VERSION != data_version )
		{
		Log(fmt("remote data version mismatch: got %d, but expected %d\n",
				data_version, DATA_FORMAT_VERSION),
				current_peer);
		CloseConnection(current_peer);
		return true;
		}

	uint32 cache_size = ntohl(args[1]);
	current_peer->cache_in->SetMaxCacheSize(cache_size);

	current_peer->got_version = true;

	delete [] c->data;
	delete c;

	current_peer = 0;
	return true;
	}


bool RemoteSerializer::ProcessConnected()
	{
	// IP and port follow.
	ChunkedIO::Chunk* c;
	READ_CHUNK_FROM_CHILD(c);

	state = MSG_NONE;

	uint32* args = (uint32*) c->data;
	uint32 host = ntohl(args[0]);	// ### Fix: only works for IPv4
	uint16 port = (uint16) ntohl(args[1]);

	if ( ! current_peer )
		{
		// The other side connected to one of our listening ports.
		current_peer = AddPeer(host, port, current_id);
		current_peer->orig = false;
		}
	else if ( current_peer->orig )
		{
		// It's a successful retry.
		current_peer->port = port;
		current_peer->accept_state = false;
		Unref(current_peer->val);
		current_peer->val = MakePeerVal(current_peer);
		}

	PeerConnected(current_peer);

	ID* descr = global_scope()->Lookup("peer_description");
	if ( ! descr )
		internal_error("peer_description not defined");

	SerialInfo info(this);
	SendID(&info, current_peer->id, *descr);

	delete [] c->data;
	delete c;

	current_peer = 0;
	return true;
	}

bool RemoteSerializer::ProcessRequestEventsMsg()
	{
	ChunkedIO::Chunk* c;
	READ_CHUNK_FROM_CHILD(c);
	state = MSG_NONE;

	if ( ! current_peer )
		return false;

	// Register new handlers.
	char* p = c->data;
	while ( p < c->data + c->len )
		{
		EventHandler* handler = event_registry->Lookup(p);
		if ( handler )
			{
			handler->AddRemoteHandler(current_peer->id);
			current_peer->handlers.append(handler);
			Log(fmt("registered for event %s", p), current_peer);
			}
		else
			Log(fmt("request for unknown event %s", p), current_peer);

		p += strlen(p) + 1;
		}

	delete [] c->data;
	delete c;

	current_peer = 0;
	return true;
	}

bool RemoteSerializer::ProcessRequestSyncMsg()
	{
	current_peer->sync = true;
	state = MSG_NONE;

	// If the other side connected to us, we send it all our synchronized
	// data.
	if ( ! current_peer->orig )
		{
		SerialInfo info(this);
		SendAllSynchronized(&info, current_peer);
		}

	Log("enabled ID synchronization", current_peer);

	current_peer = 0;
	return true;
	}

bool RemoteSerializer::ProcessHandshakeComplete()
	{
	if ( current_peer->handshake == Peer::NONE )
		current_peer->handshake = Peer::RECEIVED;
	else if ( current_peer->handshake == Peer::SENT )
		current_peer->handshake = Peer::COMPLETED;

	state = MSG_NONE;
	Log(fmt("completed handshake (state is %d)", current_peer->handshake), current_peer);
	current_peer = 0;
	return true;
	}

bool RemoteSerializer::ProcessErrorMsg()
	{
	ChunkedIO::Chunk* c;
	READ_CHUNK_FROM_CHILD(c);
	state = MSG_NONE;

	ReportError(c->data);

	delete [] c->data;
	delete c;

	current_peer = 0;
	return true;
	}

bool RemoteSerializer::ProcessStatsMsg()
	{
	ChunkedIO::Chunk* c;
	READ_CHUNK_FROM_CHILD(c);
	state = MSG_NONE;

	// Take the opportunity to log our stats, too.
	LogStats();

	// Split the concatenated child stats into indiviual log messages.
	int count = 0;
	for ( char* p = c->data; p < c->data + c->len; p += strlen(p) + 1 )
		Log(fmt("child statistics: [%d] %s", count++, p), current_peer);

	delete [] c->data;
	delete c;

	current_peer = 0;
	return true;
	}

bool RemoteSerializer::ProcessCaptureFilterMsg()
	{
	ChunkedIO::Chunk* c;
	READ_CHUNK_FROM_CHILD(c);
	state = MSG_NONE;

	if ( ! current_peer )
		return false;

	RaiseEvent(remote_capture_filter, current_peer, c->data);

	delete [] c->data;
	delete c;

	current_peer = 0;
	return true;
	}

bool RemoteSerializer::ProcessSerialization()
	{
	SetCache(current_peer->cache_in);
	UnserialInfo info(this);

	info.install_globals = current_peer->accept_state;
	info.install_conns = current_peer->accept_state;
	info.ignore_callbacks = ! current_peer->accept_state;
	info.id_policy = current_peer->accept_state ?
		UnserialInfo::CopyNewToCurrent : UnserialInfo::Keep;

	// Don't propagate accesses we're going to perform.
	ignore_accesses = true;

	int i = Unserialize(&info);
	ignore_accesses = false;

	if ( i < 0 )
		// Error
		return false;

	if ( i == 0 )
		// Interrupted read.
		return true;

	state = MSG_NONE;

	// Finished unserialization.
	current_peer = 0;
	return true;
	}

void RemoteSerializer::GotEvent(const char* name, double time,
				EventHandlerPtr event, val_list* args)
	{
	DEBUG_COMM("parent: got event");

	if ( ! current_peer )
		{
		Error("unserialized event from unknown peer");
		return;
		}

	BufferedEvent* e = new BufferedEvent;

	// Our time, not the time when the event was generated.
	//	e->time = time_t(network_time ? network_time : current_time());

	// FIXME: How do we the time when there isn't any packet source?
	e->time = time_t(time);

	e->src = current_peer->id;
	e->handler = event;
	e->args = args;

	events.append(e);
	++events_received;
	}

void RemoteSerializer::GotFunctionCall(const char* name, double time,
					Func* function, val_list* args)
	{
	DEBUG_COMM("parent: got function call");

	if ( ! current_peer )
		{
		Error("unserialized function from unknown peer");
		return;
		}

	function->Call(args);
	}

void RemoteSerializer::GotID(ID* id, Val* val)
	{
	if ( ! current_peer )
		{
		Error("unserialized id from unknown peer");
		return;
		}

	if ( streq(id->Name(), "peer_description") )
		{
		Ref(val);
		current_peer->val->Assign(4, val);
		return;
		}

	if ( id->Name()[0] == '#' )
		{
		// This is a globally unique, non-user-visible ID.

		// Only MutableVals can be bound to names starting with '#'.
		assert(val->IsMutableVal());

		// It must be already installed in the global namespace:
		// either we saw it before, or MutableVal::Unserialize()
		// installed it.
		assert(global_scope()->Lookup(id->Name()));

		// Only synchronized values can arrive here.
		assert(((MutableVal*) val)->GetProperties() & MutableVal::SYNCHRONIZED);

		DBG_LOG(DBG_COMM, "got ID %s from peer\n", id->Name());
		}
	}

void RemoteSerializer::GotConnection(Connection* c)
	{
	// Nothing else to-do. Connection will be installed automatically (if allowed).
	}

void RemoteSerializer::GotStateAccess(StateAccess* s)
	{
	ODesc d;
	DBG_LOG(DBG_COMM, "got StateAccess: %s", (s->Describe(&d), d.Description()));

	if ( ! current_peer )
		{
		Error("unserialized function from unknown peer");
		return;
		}

	s->Replay();

	delete s;
	}

void RemoteSerializer::GotTimer(Timer* s)
	{
	run_time("RemoteSerializer::GotTimer not implemented");
	}

void RemoteSerializer::GotPacket(Packet* p)
	{
	sessions->NextPacket(p->time, p->hdr, p->pkt, p->hdr_size, 0);
	delete p;
	}

void RemoteSerializer::Log(const char* msg, Peer* peer)
	{
	const int BUFSIZE = 1024;
	char buffer[BUFSIZE];

	int len = snprintf(buffer, sizeof(buffer), "%u (parent) ", time(0));
	int start = len;

	if ( peer )
		len += snprintf(buffer + len, sizeof(buffer) - len,
				"[#%d/%s:%d] ", peer->id, ip2a(peer->ip),
				peer->port);

	len += snprintf(buffer + len, sizeof(buffer) - len, "%s", msg);
	write(error_fd, buffer, len);
	write(error_fd, "\n", 1);

	DEBUG_COMM(fmt("parent: %s", buffer + start));
	}

void RemoteSerializer::RaiseEvent(EventHandlerPtr event, Peer* peer, const char* arg)
	{
	val_list* vl = new val_list;
	PeerID id;

	if ( peer )
		{
		Ref(peer->val);
		vl->append(peer->val);
		id = peer->id;
		}
	else
		{
		Ref(mgr.GetLocalPeerVal());
		vl->append(mgr.GetLocalPeerVal());
		id = PEER_LOCAL;
		}

	if ( arg )
		vl->append(new StringVal(arg));

	// If we only have remote sources, the network time
	// will not increase as long as no peers are connected.
	// Therefore, we send these events immediately.
	mgr.Dispatch(new Event(event, vl, id));
	}

void RemoteSerializer::LogStats()
	{
	char buffer[512];
	assert(io);
	io->Stats(buffer, 512);
	Log(fmt("parent statistics: %s events=%lu/%lu",
		buffer, events_received, events_sent));
	}

RecordVal *RemoteSerializer::GetPeerVal(PeerID id)
	{
	Peer* peer = LookupPeer(id, true);
	if ( ! peer )
		return 0;

	Ref(peer->val);
	return peer->val;
	}

void RemoteSerializer::ChildDied()
	{
	Log("child died");
	child_pid = 0;
	}

bool RemoteSerializer::SendToChild(char type, Peer* peer, char* str, int len)
	{
	DEBUG_COMM(fmt("parent: (->child) %s (#%d, %s)", msgToStr(type), peer ? peer->id : PEER_NONE, str));

	if ( ! child_pid )
		return false;

	if ( sendToIO(io, type, peer ? peer->id : PEER_NONE, str, len) )
		return true;

	if ( io->Eof() )
		ChildDied();

	return false;
	}

bool RemoteSerializer::SendToChild(char type, Peer* peer, int nargs, ...)
	{
	va_list ap;

	if ( ! child_pid )
		return false;

#ifdef DEBUG
	va_start(ap, nargs);
	DEBUG_COMM(fmt("parent: (->child) %s (#%d,%s)",
			msgToStr(type), peer ? peer->id : PEER_NONE, fmt_uint32s(nargs, ap)));
	va_end(ap);
#endif

	va_start(ap, nargs);
	bool ret = sendToIO(io, type, peer ? peer->id : PEER_NONE, nargs, ap);
	va_end(ap);

	if ( ret )
		return true;

	if ( io->Eof() )
		ChildDied();

	return false;
	}

bool RemoteSerializer::SendToChild(ChunkedIO::Chunk* c)
	{
	DEBUG_COMM(fmt("parent: (->child) chunk of size %d", c->len));

	if ( ! child_pid )
		return false;

	if ( sendToIO(io, c) )
		return true;

	if ( io->Eof() )
		ChildDied();

	return false;
	}

bool RemoteSerializer::IsActive()
	{
	if ( listening )
		return true;

	loop_over_list(peers, i)
		if ( peers[i]->state == Peer::PENDING ||
		     peers[i]->state == Peer::CONNECTED )
			return true;

	return false;
	}


const char* const* RemoteSerializer::GetBuiltins() const
	{
	static const char* builtins[] = { "connect", "listen", 0 };
	return builtins;
	}

void RemoteSerializer::ReportError(const char* msg)
	{
	RaiseEvent(remote_connection_error, current_peer, msg);
	Log(msg, current_peer);
	}

////////////////////////////

// If true (set by signal handler), we will log some stats to parent.
static bool log_stats = false;

// How often stats are sent (in seconds).
// Perhaps we should make this configurable...
const int STATS_INTERVAL = 60;

static RETSIGTYPE sig_handler(int signo)
	{
	// SIGALRM is the only one we get.
	log_stats = true;
	}

SocketComm::SocketComm(int arg_error_fd)
	{
	io = 0;
	error_fd = arg_error_fd;

	// We start the ID counter high so that IDs assigned by us
	// (hopefully) don't conflict with those of our parent.
	id_counter = 10000;
	parent_peer = 0;
	parent_state = MSG_NONE;

	listen_fd_clear = -1;
	listen_fd_ssl = -1;
	listen_next_try = 0;

	// We don't want to use the signal handlers of our parent.
	(void) setsignal(SIGTERM, SIG_DFL);
	(void) setsignal(SIGINT, SIG_DFL);
	(void) setsignal(SIGUSR1, SIG_DFL);
	(void) setsignal(SIGUSR2, SIG_DFL);
	(void) setsignal(SIGCONT, SIG_DFL);
	(void) setsignal(SIGCHLD, SIG_DFL);

	(void) setsignal(SIGALRM, sig_handler);
	alarm(STATS_INTERVAL);
	}

SocketComm::~SocketComm()
	{
	loop_over_list(peers, i)
		delete peers[i]->io;

	delete io;
	close(error_fd);
	close(listen_fd_clear);
	close(listen_fd_ssl);
	}

void SocketComm::Run()
	{
	while ( true )
		{
		// Logging signaled?
		if ( log_stats )
			LogStats();

		// Build FDSets for select.
		fd_set fd_read, fd_write, fd_except;

		FD_ZERO(&fd_read);
		FD_ZERO(&fd_write);
		FD_ZERO(&fd_except);

		int max_fd = 0;

		FD_SET(io->Fd(), &fd_read);
		max_fd = io->Fd();

		loop_over_list(peers, i)
			{
			if ( peers[i]->connected )
				{
				FD_SET(peers[i]->io->Fd(), &fd_read);
				if ( peers[i]->io->Fd() > max_fd )
					max_fd = peers[i]->io->Fd();
				}
			else
				{
				if ( peers[i]->next_try > 0 && time(0) > peers[i]->next_try )
					// Try reconnect.
					Connect(peers[i]);
				}
			}

		if ( listen_next_try && time(0) > listen_next_try  )
			Listen(listen_if, listen_port, listen_ssl);

		if ( listen_fd_clear >= 0 )
			{
			FD_SET(listen_fd_clear, &fd_read);
			if ( listen_fd_clear > max_fd )
				max_fd = listen_fd_clear;
			}

		if ( listen_fd_ssl >= 0 )
			{
			FD_SET(listen_fd_ssl, &fd_read);
			if ( listen_fd_ssl > max_fd )
				max_fd = listen_fd_ssl;
			}

		// We cannot rely solely on select() as the there may
		// be some data left in our input/output queues. So, we use
		// a small timeout for select and check for data
		// manually afterwards.

		// FIXME: Fine-tune this (timeouts, flush, etc.)
		struct timeval small_timeout;
		small_timeout.tv_sec = 0;
		small_timeout.tv_usec = io->CanWrite() ? 1 : 10;

		if ( select(max_fd + 1, &fd_read, &fd_write, &fd_except,
				&small_timeout) < 0 )
			// Ignore errors for now.
			continue;

		if ( io->CanRead() )
			ProcessParentMessage();

		io->Flush();

		loop_over_list(peers, j)
			{
			// We have to be careful here as the peer may
			// be removed when an error occurs.
			Peer* current = peers[j];
			while ( j < peers.length() && peers[j] == current &&
				current->connected && current->io->CanRead() )
				{
				ProcessRemoteMessage(current);
				}
			}

		if ( listen_fd_clear >= 0 && FD_ISSET(listen_fd_clear, &fd_read) )
			AcceptConnection(listen_fd_clear);

		if ( listen_fd_ssl >= 0 && FD_ISSET(listen_fd_ssl, &fd_read) )
			AcceptConnection(listen_fd_ssl);

		}
	}

bool SocketComm::ProcessParentMessage()
	{
	switch ( parent_state ) {
	case MSG_NONE:
		{
		// CMsg follows.
		ChunkedIO::Chunk* c;
		if ( ! io->Read(&c) )
			{
			if ( io->Eof() )
				{
				Log("parent died");
				Kill();
				}

			Error(fmt("can't read parent's cmsg: %s",
					io->Error()), 0, true);
			return false;
			}

		if ( ! c )
			return true;

		CMsg* msg = (CMsg*) c->data;
		parent_peer = LookupPeer(ntohl(msg->peer), false);
		parent_id = ntohl(msg->peer);

		DEBUG_COMM(fmt("child: %s from parent; peer is #%d", msgToStr(msg->type), ntohl(msg->peer)));

		switch ( msg->type ) {
		case MSG_LISTEN:
			ProcessListen();
			break;

		case MSG_LISTEN_STOP:
			if ( listen_fd_ssl >= 0 )
				close(listen_fd_ssl);

			if ( listen_fd_clear >= 0 )
				close(listen_fd_clear);

			listen_fd_clear = listen_fd_ssl = -1;
			Log("stopped listening");
			break;


		case MSG_CLOSE:
			if ( parent_peer && parent_peer->connected )
				CloseConnection(parent_peer, false);
			break;

		case MSG_CLOSE_ALL:
			loop_over_list(peers, i)
				{
				if ( peers[i]->connected )
					CloseConnection(peers[i], false);
				}

		case MSG_REQUEST_SYNC:
		case MSG_HANDSHAKE_COMPLETE:
			// No argument block follows.
			if ( parent_peer && parent_peer->connected )
				{
				DEBUG_COMM("child: forwarding message...");
				if ( ! SendToPeer(parent_peer, msg->type, 0) )
					return false;
				}

			parent_state = MSG_NONE;
			break;

		default:
			parent_state = msg->type;
		}

		delete [] c->data;
		delete c;
		break;
		}

	case MSG_CONNECT_TO:
		ProcessConnectTo();
		break;

	case MSG_REQUEST_EVENTS:
	case MSG_SERIAL:
	case MSG_CAPTURE_FILTER:
	case MSG_VERSION:
		{
		// One argument block follows.
		ChunkedIO::Chunk* c;
		READ_CHUNK(io, c, (Log("parent died"), Kill()));

		char state = parent_state;
		parent_state = MSG_NONE;

		if ( parent_peer && parent_peer->connected )
			{
			DEBUG_COMM("child: forwarding message...");

			if ( ! SendToPeer(parent_peer, state, 0) )
				return false;

			if ( ! SendToPeer(parent_peer, c) )
				return false;
			}
		else
			{
#ifdef DEBUG
			if ( parent_peer )
				DEBUG_COMM(fmt("child: not connected to #%d", parent_id));
#endif
			delete [] c->data;
			delete c;
			}

		break;
		}

	default:
		internal_error("ProcessParentMessage: unexpected state");
	}

	return true;
	}

bool SocketComm::ProcessConnectTo()
	{
	ChunkedIO::Chunk* c;
	READ_CHUNK(io, c, (Log("parent died"), Kill()));
	uint32* args = (uint32*) c->data;

	Peer* peer = new Peer;
	peer->id = ntohl(args[0]);
	peer->ip = ntohl(args[1]);
	peer->port = ntohl(args[2]);
	peer->retry = ntohl(args[3]);
	peer->ssl = ntohl(args[4]);

	Connect(peer);

	delete [] c->data;
	delete c;

	parent_state = MSG_NONE;
	return true;
	}

bool SocketComm::ProcessListen()
	{
	ChunkedIO::Chunk* c;
	READ_CHUNK(io, c, (Log("parent died"), Kill()));
	uint32* args = (uint32*) c->data;

	uint32 addr = ntohl(args[0]);
	uint16 port = uint16(ntohl(args[1]));
	uint32 ssl = ntohl(args[2]);

	delete [] c->data;
	delete c;

	return Listen(addr, port, ssl);
	}

bool SocketComm::ProcessRemoteMessage(Peer* peer)
	{
	assert(peer);

	switch ( peer->state ) {
	case MSG_NONE:
		{ // CMsg follows
		ChunkedIO::Chunk* c;
		READ_CHUNK(peer->io, c,
			(Error(io->Error(), peer), CloseConnection(peer, true)))

		CMsg* msg = (CMsg*) c->data;

		DEBUG_COMM(fmt("child: %s from peer #%d", msgToStr(msg->type), peer->id));

		switch ( msg->type ) {
		case MSG_HANDSHAKE_COMPLETE:
		case MSG_REQUEST_SYNC:
			// No further argument block.
			if ( ! SendToParent(msg->type, peer, 0) )
				{
				delete [] c->data;
				delete c;
				return false;
				}
			break;

		default:
			peer->state = msg->type;
			break;
		}

		delete [] c->data;
		delete c;

		break;
		}

	case MSG_REQUEST_EVENTS:
	case MSG_SERIAL:
	case MSG_CAPTURE_FILTER:
	case MSG_VERSION:
		{
		// Messages with one further argument block which we simply
		// forward to our parent.
		ChunkedIO::Chunk* c;
		READ_CHUNK(peer->io, c,
			(Error(io->Error(), peer), CloseConnection(peer, true)));

		char state = peer->state;
		peer->state = MSG_NONE;

		DEBUG_COMM("child: forwarding message to parent...");

		if ( ! SendToParent(state, peer, 0) )
			return false;

		if ( ! SendToParent(c) )
			return false;

		io->Flush(); // FIXME: Needed?

		break;
		}

	default:
		internal_error("ProcessRemoteMessage: unexpected state");
	}

	return true;
	}

bool SocketComm::Connect(Peer* peer)
	{
	struct sockaddr_in server;

	int sockfd = socket(PF_INET, SOCK_STREAM, 0);
	if ( sockfd < 0 )
		{
		Error(fmt("can't create socket, %s", strerror(errno)), 0, false);
		return false;
		}

	bzero(&server, sizeof(server));
	server.sin_family = AF_INET;
	server.sin_port = htons(peer->port);
	server.sin_addr.s_addr = htonl(peer->ip);

	bool connected = true;

	if ( connect(sockfd, (sockaddr*) &server, sizeof(server)) < 0 )
		{
		Error(fmt("connect failed: %s", strerror(errno)), peer);
		close(sockfd);
		connected = false;
		}

	if ( ! (connected || peer->retry) )
		{
		CloseConnection(peer, false);
		return false;
		}

	Peer* existing_peer = LookupPeer(peer->id, false);
	if ( existing_peer )
		{
		*existing_peer = *peer;
		peer = existing_peer;
		}
	else
		peers.append(peer);

	peer->connected = connected;
	peer->next_try = connected ? 0 : time(0) + peer->retry;
	peer->state = MSG_NONE;
	peer->io = 0;

	if ( connected )
		{
		if ( peer->ssl )
			{
#ifdef USE_OPENSSL
			peer->io = new ChunkedIOSSL(sockfd, false);
#else
			run_time("SSL connection requested, but SSL support not compiled in");
			CloseConnection(peer, false);
			return 0;
#endif
			}
		else
			peer->io = new ChunkedIOFd(sockfd, "child->peer");

		if ( ! peer->io->Init() )
			{
			Error(fmt("can't init peer io: %s",
					peer->io->Error()), peer, false);
			CloseConnection(peer, true);
			return 0;
			}
		}

	if ( connected )
		{
		Log("Connected", peer);
		if ( ! SendToParent(MSG_CONNECTED, peer, 2, peer->ip, peer->port) )
			return false;
		}

	return connected;
	}

bool SocketComm::CloseConnection(Peer* peer, bool reconnect)
	{
	if ( ! SendToParent(MSG_CLOSE, peer, 0) )
		return false;

	Log("connection closed", peer);

	if ( ! peer->retry || ! reconnect )
		{
		peers.remove(peer);
		delete peer->io; // This will close the fd.
		delete peer;
		}
	else
		{
		delete peer->io; // This will close the fd.
		peer->io = 0;
		peer->connected = false;
		peer->next_try = time(0) + peer->retry;
		}

	if ( parent_peer == peer )
		{
		parent_peer = 0;
		parent_id = RemoteSerializer::PEER_NONE;
		}

	return true;
	}

bool SocketComm::Listen(uint32 ip, uint16 port, bool expect_ssl)
	{
	int* listen_fd = expect_ssl ? &listen_fd_ssl : &listen_fd_clear;

	if ( *listen_fd >= 0 )
		close(*listen_fd);

	struct sockaddr_in server;

	*listen_fd = socket(PF_INET, SOCK_STREAM, 0);
	if ( *listen_fd < 0 )
		{
		Error(fmt("can't create listen socket, %s",
				strerror(errno)), 0, false);
		return false;
		}

	// Set SO_REUSEADDR.
	int turn_on = 1;
	if ( setsockopt(*listen_fd, SOL_SOCKET, SO_REUSEADDR,
			&turn_on, sizeof(turn_on)) < 0 )
		{
		Error(fmt("can't set SO_REUSEADDR, %s",
				strerror(errno)), 0, false);
		return false;
		}

	bzero(&server, sizeof(server));
	server.sin_family = AF_INET;
	server.sin_port = htons(port);
	server.sin_addr.s_addr = htonl(ip);

	if ( bind(*listen_fd, (sockaddr*) &server, sizeof(server)) < 0 )
		{
		Error(fmt("can't bind to port, %s", strerror(errno)), 0, false);
		*listen_fd = -1;

		if ( errno == EADDRINUSE )
			{
			listen_if = ip;
			listen_port = port;
			listen_ssl = expect_ssl;
			// FIXME: Make this timeout configurable.
			listen_next_try = time(0) + 30;
			}
		return false;
		}

	if ( listen(*listen_fd, 50) < 0 )
		{
		Error(fmt("can't listen, %s", strerror(errno)), 0, false);
		return false;
		}

	listen_next_try = 0;
	Log(fmt("listening on %s:%d (%s)", ip2a(ip), port, expect_ssl ? "ssl" : "clear"));
	return true;
	}

bool SocketComm::AcceptConnection(int fd)
	{
	sockaddr_in client;
	socklen_t len = sizeof(client);

	int clientfd = accept(fd, (sockaddr*) &client, &len);
	if ( clientfd < 0 )
		{
		Error(fmt("accept failed, %s %d",
				strerror(errno), errno), 0, false);
		return false;
		}

	Peer* peer = new Peer;
	peer->id = id_counter++;
	peer->ip = ntohl(client.sin_addr.s_addr);
	peer->port = ntohs(client.sin_port);
	peer->connected = true;
	peer->ssl = (fd == listen_fd_ssl);

#ifdef USE_OPENSSL
	if ( peer->ssl )
		peer->io = new ChunkedIOSSL(clientfd, true);
	else
		peer->io = new ChunkedIOFd(clientfd, "child->peer");
#else
	assert(! peer->ssl);
	peer->io = new ChunkedIOFd(clientfd, "child->peer");
#endif

	if ( ! peer->io->Init() )
		{
		Error(fmt("can't init peer io: %s",
				  peer->io->Error()), peer, false);
		return false;
		}

	peers.append(peer);

	Log(fmt("accepted %s connection", peer->ssl ? "SSL" : "clear"), peer);

	if ( ! SendToParent(MSG_CONNECTED, peer, 2, peer->ip, peer->port) )
		return false;

	return true;
	}

void SocketComm::Error(const char* msg, Peer* peer, bool kill_me)
	{
	Log(msg, peer);

	// If a remote peer causes an error, we shutdown the connection
	// as resynchronizing is in general not possible. But we may
	// try again later.
	if ( peer && peer->connected )
		CloseConnection(peer, true);

	if ( kill_me )
		Kill();
	}

void SocketComm::Log(const char* msg, Peer* peer)
	{
	const int BUFSIZE = 1024;
	char buffer[BUFSIZE];

	int len = snprintf(buffer, sizeof(buffer), "%lu (child)  ", time(0));
	int start = len;

	if ( peer )
		len += snprintf(buffer + len, sizeof(buffer) - len,
				"[#%d/%s:%d] ", peer->id, ip2a(peer->ip),
				peer->port);

	len += snprintf(buffer + len, sizeof(buffer) - len, "%s", msg);

	write(error_fd, buffer, len);
	write(error_fd, "\n", 1);
	DEBUG_COMM(fmt("child: %s", buffer + start));
	}

bool SocketComm::Error(const char* msg, Peer* peer)
	{
	return SendToParent(MSG_ERROR, peer, copy_string(msg));
	}

void SocketComm::Kill()
	{
	Log("terminating");

	close(error_fd);
	close(listen_fd_clear);
	close(listen_fd_ssl);

	kill(getpid(), SIGTERM);

	while ( 1 )
		; // loop until killed
	}

SocketComm::Peer* SocketComm::LookupPeer(RemoteSerializer::PeerID id, bool only_if_connected)
	{
	loop_over_list(peers, i)
		if ( peers[i]->id == id )
			return ! only_if_connected ||
				peers[i]->connected ? peers[i] : 0;
	return 0;
	}

bool SocketComm::LogStats()
	{
	if ( ! peers.length() )
		return true;

	// Concat stats of all peers into single buffer.
	char* buffer = new char[peers.length() * 512];
	int pos = 0;

	loop_over_list(peers, i)
		{
		if ( peers[i]->connected )
			peers[i]->io->Stats(buffer+pos, 512);
		else
			strcpy(buffer+pos, "not connected");
		pos += strlen(buffer+pos) + 1;
		}

	// Send it.
	if ( ! SendToParent(MSG_STATS, 0, buffer, pos) )
		return false;

	log_stats = false;
	alarm(STATS_INTERVAL);
	return true;
	}

bool SocketComm::SendToParent(char type, Peer* peer, const char* str, int len)
	{
#ifdef DEBUG
	// str  may already by constructed with fmt()
	const char* tmp = copy_string(str);
	DEBUG_COMM(fmt("child: (->parent) %s (#%d, %s)", msgToStr(type), peer ? peer->id : RemoteSerializer::PEER_NONE, tmp));
	delete [] tmp;
#endif
	if ( sendToIO(io, type, peer ? peer->id : RemoteSerializer::PEER_NONE, str, len) )
		return true;

	if ( io->Eof() )
		{
		Log("parent died");
		Kill();
		}

	return false;
	}

bool SocketComm::SendToParent(char type, Peer* peer, int nargs, ...)
	{
	va_list ap;

#ifdef DEBUG
	va_start(ap,nargs);
	DEBUG_COMM(fmt("child: (->parent) %s (#%d,%s)", msgToStr(type), peer->id, fmt_uint32s(nargs, ap)));
	va_end(ap);
#endif

	va_start(ap, nargs);
	bool ret = sendToIO(io, type, peer->id, nargs, ap);
	va_end(ap);

	if ( ret )
		return true;

	if ( io->Eof() )
		{
		Log("parent died");
		Kill();
		}

	return false;
	}

bool SocketComm::SocketComm::SendToParent(ChunkedIO::Chunk* c)
	{
	DEBUG_COMM(fmt("child: (->parent) chunk of size %d", c->len));
	if ( sendToIO(io, c) )
		return true;

	if ( io->Eof() )
		{
		Log("parent died");
		Kill();
		}

	return false;
	}

bool SocketComm::SendToPeer(Peer* peer, char type, const char* str, int len)
	{
#ifdef DEBUG
	// str  may already by constructed with fmt()
	const char* tmp = copy_string(str);
	DEBUG_COMM(fmt("child: (->peer) %s to #%d (%s)", msgToStr(type), peer->id, tmp));
	delete [] tmp;
#endif

	if ( ! sendToIO(peer->io, type, RemoteSerializer::PEER_NONE, str, len) )
		{
		Error(fmt("child: write error %s", io->Error()));
		CloseConnection(peer, true);
		return false;
		}

	return true;
	}

bool SocketComm::SendToPeer(Peer* peer, char type, int nargs, ...)
	{
	va_list ap;

#ifdef DEBUG
	va_start(ap,nargs);
	DEBUG_COMM(fmt("child: (->peer) %s to #%d (%s)",
			msgToStr(type), peer->id, fmt_uint32s(nargs, ap)));
	va_end(ap);
#endif

	va_start(ap, nargs);
	bool ret = sendToIO(peer->io, type, RemoteSerializer::PEER_NONE, nargs, ap);
	va_end(ap);

	if ( ! ret )
		{
		Error(fmt("child: write error %s", io->Error()));
		CloseConnection(peer, true);
		return false;
		}

	return true;
	}

bool SocketComm::SendToPeer(Peer* peer, ChunkedIO::Chunk* c)
	{
	DEBUG_COMM(fmt("child: (->peer) chunk of size %d to #%d", c->len, peer->id));
	if ( ! sendToIO(peer->io, c) )
		{
		Error(fmt("child: write error %s", io->Error()));
		CloseConnection(peer, true);
		return false;
		}

	return true;
	}

