// $Id: RemoteSerializer.h,v 1.7 2005/02/09 08:19:03 vern Exp $
//
// Communication between two Bro's.

#ifndef REMOTE_SERIALIZER
#define REMOTE_SERIALIZER

#include "Dict.h"
#include "List.h"
#include "Serializer.h"
#include "IOSource.h"

// All IP arguments are in host byte-order.
// FIXME: Change this to network byte order

// This class handles the communication done in Bro's main loop.
class RemoteSerializer : public Serializer, public IOSource {
public:
	RemoteSerializer();
	virtual ~RemoteSerializer();

	// Initialize the remote serializer (calling this will fork).
	void Init();

	// FIXME: Use SourceID directly (or rename everything to Peer*).
	typedef SourceID PeerID;
	static const PeerID PEER_LOCAL = SOURCE_LOCAL;
	static const PeerID PEER_NONE = SOURCE_LOCAL;

	// Connect to host (returns PEER_NONE on error).
	PeerID Connect(addr_type ip, uint16 port, double retry, bool use_ssl);

	// Request all events matching pattern from remote side.
	bool RequestEvents(PeerID peer, RE_Matcher* pattern);

	// Request synchronization of IDs with remote side.
	bool RequestSync(PeerID peer);

	// Sets flag whether we're accepting state from this peer
	// (default: yes).
	bool SetAcceptState(PeerID peer, bool accept);

	// Signal the other side that we have finished our part of
	// the initial handshake.
	bool CompleteHandshake(PeerID peer);

	// Start to listen.
	bool Listen(addr_type ip, uint16 port, bool expect_ssl);

	// Stop it.
	bool StopListening();

	// Send the event/function call (only if handshake completed).
	bool SendCall(SerialInfo* info, PeerID peer, const char* name, val_list* vl);

	// Broadcasts the access (only if handshake completed).
	bool SendAccess(SerialInfo* info, const StateAccess& access);

	// Sends ID.
	bool SendID(SerialInfo* info, PeerID peer, const ID& id);

	// Sends the internal connection state.
	bool SendConnection(SerialInfo* info, PeerID peer, const Connection& c);

	// Send capture filter.
	bool SendCaptureFilter(PeerID peer, const char* filter);

	// Send packet.
	bool SendPacket(SerialInfo* info, PeerID peer, const Packet& p);

	// Registers the ID to be &synchronized.
	void Register(ID* id);
	void Unregister(ID* id);

	// Check for incoming events and queue them.
	bool Poll(bool may_block);

	// Returns the corresponding record (already ref'ed).
	RecordVal* GetPeerVal(PeerID id);

	// Log some statistics.
	void LogStats();

	// Return a 0-terminated array of built-in functions which,
	// when referenced, trigger the remote serializer's initialization.
	const char* const* GetBuiltins() const;

	// Overidden from IOSource:
	virtual void GetFds(int* read, int* write, int* except);
	virtual double NextTimestamp();
	virtual void Process();

protected:
	// Maximum size of serialization caches.
	static const unsigned int MAX_CACHE_SIZE = 1000;

	declare(PList, EventHandler);
	typedef PList(EventHandler) handler_list;

	struct Peer {
		PeerID id; // Unique ID (non-zero) per peer.

		// ### Fix: currently, we only work for IPv4.
		// addr_type ip;
		uint32 ip;

		uint16 port;
		handler_list handlers;
		RecordVal* val;		// Record of type event_source.
		SerializationCache* cache_in;	// One cache for each direction.
		SerializationCache* cache_out;

		// State of the connection to the peer.
		enum { INIT, PENDING, CONNECTED, CLOSED } state;

		// State of the handshake.
		enum { NONE, SENT, RECEIVED, COMPLETED } handshake;

		bool orig;	// True if we connected to the peer.
		bool accept_state;	// True if we accept state from peer.
		bool sync;	// True if peer wants synchronized IDs.
		bool got_version;	// True if version handshake succeeded.
	};

	void Log(const char* msg, Peer* peer = 0);

	virtual void ReportError(const char* msg);

	virtual void GotEvent(const char* name, double time,
				EventHandlerPtr event, val_list* args);
	virtual void GotFunctionCall(const char* name, double time,
				Func* func, val_list* args);
	virtual void GotID(ID* id, Val* val);
	virtual void GotStateAccess(StateAccess* s);
	virtual void GotTimer(Timer* t);
	virtual void GotConnection(Connection* c);
	virtual void GotPacket(Packet* packet);

	void Fork();

	bool ProcessConnected();
	bool ProcessSerialization();
	bool ProcessRequestEventsMsg();
	bool ProcessRequestSyncMsg();
	bool ProcessVersionMsg();
	bool ProcessErrorMsg();
	bool ProcessStatsMsg();
	bool ProcessCaptureFilterMsg();
	bool ProcessHandshakeComplete();

	Peer* AddPeer(uint32 ip, uint16 port, PeerID id = PEER_NONE);
	Peer* LookupPeer(PeerID id, bool only_if_connected);
	void RemovePeer(Peer* peer);
	bool IsConnectedPeer(PeerID id);
	void PeerDisconnected(Peer* peer);
	void PeerConnected(Peer* peer);
	RecordVal* MakePeerVal(Peer* peer);
	bool IsActive();

	bool CloseConnection(Peer* peer);

	bool SendAllSynchronized(SerialInfo* info, Peer* peer);
	bool SendAccess(SerialInfo* info, PeerID peer, const StateAccess& access);

	void UnregisterHandlers(Peer* peer);
	void RaiseEvent(EventHandlerPtr event, Peer* peer, const char* arg = 0);

	void ChildDied();

	// Communication helpers
	bool SendToChild(char type, Peer* peer, char* str, int len = -1);
	bool SendToChild(char type, Peer* peer, int nargs, ...); // can send uints32 only
	bool SendToChild(ChunkedIO::Chunk* c);

private:
	char state;	// Current communication state; see MSG_* in RemoteSerializer.cc.
	Peer* current_peer;
	PeerID current_id;

	id_list sync_ids;

	// FIXME: Check which of these are necessary...
	bool initialized;
	bool listening;
	bool ignore_accesses;
	int error_fd;
	PeerID id_counter;	// Keeps track of assigned IDs.

	declare(PList, Peer);
	typedef PList(Peer) peer_list;
	peer_list peers;

	// Event buffer
	struct BufferedEvent {
		time_t time;
		PeerID src;
		EventHandlerPtr handler;
		val_list* args;
	};

	declare(PList, BufferedEvent);
	typedef PList(BufferedEvent) EventQueue;
	EventQueue events;

	// Some stats
	unsigned long events_received;
	unsigned long events_sent;
	};

// This class handles the communication done in the forked child.
class SocketComm {
public:
	SocketComm(int errorfd);
	~SocketComm();

	void SetParentIO(ChunkedIO* arg_io)	{ io = arg_io; }

	void Run();	// does not return

	// Log some statistics (via pipe to parent).
	bool LogStats();

protected:
	struct Peer {
		Peer()
			{
			id = 0;
			io = 0;
			ip = 0;
			port = 0;
			state = 0;
			connected = false;
			ssl = false;
			retry = 0;
			next_try = 0;
			}

		RemoteSerializer::PeerID id;
		ChunkedIO* io;
		uint32 ip;
		uint16 port;
		char state;
		bool connected;
		bool ssl;
		// If we get disconnected, reconnect after this many seconds.
		int retry;
		// Time of next connection attempt (0 if none).
		time_t next_try;
	};

	bool Listen(uint32 ip, uint16 port, bool expect_ssl);
	bool AcceptConnection(int listen_fd);
	bool Connect(Peer* peer);
	bool CloseConnection(Peer* peer, bool reconnect);

	Peer* LookupPeer(RemoteSerializer::PeerID id, bool only_if_connected);

	bool ProcessRemoteMessage(Peer* peer);
	bool ProcessParentMessage();

	bool ProcessListen();
	bool ProcessConnectTo();

	void Log(const char* msg, Peer* peer = 0);
	bool Error(const char* msg, Peer* peer = 0);

	// If peer is not nil, the peer caused the error and the connection
	// to it will be closed. If kill is true, this is a fatal error,
	// and the process will kill itself.
	void Error(const char* msg, Peer* peer, bool kill);

	// Kill the current process.
	void Kill();

	// Communication helpers.
	bool SendToParent(char type, Peer* peer, const char* str, int len = -1);
	bool SendToParent(char type, Peer* peer, int nargs, ...); // can send uints32 only
	bool SendToParent(ChunkedIO::Chunk* c);
	bool SendToPeer(Peer* peer, char type, const char* str, int len = -1);
	bool SendToPeer(Peer* peer, char type, int nargs, ...); // can send uints32 only
	bool SendToPeer(Peer* peer, ChunkedIO::Chunk* c);

	// Peers we are communicating with:
	declare(PList, Peer);
	typedef PList(Peer) peer_list;

	RemoteSerializer::PeerID id_counter;
	peer_list peers;

	ChunkedIO* io;	// I/O to parent
	char parent_state;

	Peer* parent_peer;
	RemoteSerializer::PeerID parent_id;

	int error_fd;
	int listen_fd_clear;
	int listen_fd_ssl;

	// If the port we're trying to bind to is already in use, we will retry
	// it regularly.
	uint32 listen_if;	// Fix: only supports IPv4
	uint16 listen_port;
	bool listen_ssl;
	time_t listen_next_try;
};

extern RemoteSerializer* remote_serializer;

#endif
