// $Id: SerialObj.cc,v 1.2 2004/09/17 03:52:27 vern Exp $

#include "SerialObj.h"
#include "Serializer.h"

TransientID::ID TransientID::counter = 0;

SerialObj::FactoryMap* SerialObj::factories = 0;
SerialObj::ClassNameMap* SerialObj::names = 0;

SerialObj* SerialObj::Instantiate(SerialType type)
	{
	FactoryMap::iterator f = factories->find(type);
	if ( f != factories->end() )
		return (SerialObj*) (*f->second)();

	run_time(fmt("Unknown object type 0x%08x", type));
	return 0;
	}

const char* SerialObj::ClassName(SerialType type)
	{
	ClassNameMap::iterator f = names->find(type);
	if ( f != names->end() )
		return f->second;

	run_time(fmt("Unknown object type 0x%08x", type));
	return "<no-class-name>";
	}

void SerialObj::Register(SerialType type, FactoryFunc f, const char* name)
	{
	if ( ! factories )
		{
		factories = new FactoryMap;
		names = new ClassNameMap;
		}

	FactoryMap::iterator i = factories->find(type);
	if ( i != factories->end() )
		internal_error(fmt("SerialType 0x%08x registered twice", type));

	(*factories)[type] = f;
	(*names)[type] = name;

	}

bool SerialObj::Serialize(SerialInfo* info) const
	{
	assert(info);

	SerializationCache::PermanentID pid = SerializationCache::NONE;

	const TransientID* tid = GetTID();
	if ( tid && info->cache )
		{
		pid = info->s->Cache()->Lookup(*tid);

		if ( pid != SerializationCache::NONE )
			{
			DBG_LOG(DBG_SERIAL, "%s [%p, ref pid %d, tid %d]", __PRETTY_FUNCTION__, this, pid, tid->Value() );

			DBG_LOG(DBG_SERIAL, "-- Caching");
			DBG_PUSH(DBG_SERIAL);
			if ( ! (SERIALIZE(false) && SERIALIZE(pid)) )
				{
				DBG_POP(DBG_SERIAL);
				return false;
				}

			DBG_POP(DBG_SERIAL);
			return true;
			}

		pid = info->s->Cache()->Register(this);

		DBG_LOG(DBG_SERIAL, "%s [%p, new pid %d, tid %d]", __PRETTY_FUNCTION__, this, pid, tid->Value() );
		DBG_LOG(DBG_SERIAL, "-- Caching");
		DBG_PUSH(DBG_SERIAL);

		if ( ! (SERIALIZE(true) && SERIALIZE(pid)) )
			{
			DBG_POP(DBG_SERIAL);
			return false;
			}

		DBG_POP(DBG_SERIAL);
		}

	else
		{
		DBG_LOG(DBG_SERIAL, "%s [%p, no caching]", __PRETTY_FUNCTION__, this);
		}

	info->type = SER_NONE;

	DBG_PUSH(DBG_SERIAL);
	bool ret = DoSerialize(info);
	DBG_POP(DBG_SERIAL);

#ifdef DEBUG
	if ( debug_logger.IsEnabled(DBG_SERIAL) && IsBroObj(serial_type) )
		{
		ODesc desc(DESC_READABLE);
		((BroObj*)this)->Describe(&desc);
		DBG_LOG(DBG_SERIAL, "-- Desc: %s", desc.Description());
		}
#endif

	return ret;
	}

SerialObj* SerialObj::Unserialize(UnserialInfo* info, SerialType type)
	{
	SerializationCache::PermanentID pid = SerializationCache::NONE;

	DBG_LOG(DBG_SERIAL, "%s", __PRETTY_FUNCTION__);

	if ( info->cache )
		{
		bool full_obj;

		DBG_LOG(DBG_SERIAL, "-- Caching");
		DBG_PUSH(DBG_SERIAL);
		if ( ! (UNSERIALIZE(&full_obj) && UNSERIALIZE(&pid)) )
			{
			DBG_POP(DBG_SERIAL);
			return false;
			}
		DBG_POP(DBG_SERIAL);

		DBG_LOG(DBG_SERIAL, "-- [%s pid %d]", full_obj ? "obj" : "ref", pid);

		if ( ! full_obj )
			{
			// FIXME: Yet another const_cast to check eventually...
			SerialObj* obj = const_cast<SerialObj *>(info->s->Cache()->Lookup(pid));
			if ( obj )
				{
				if ( obj->IsBroObj() )
					Ref((BroObj*)obj);
				return obj;
				}

			info->s->Error(fmt("unknown object %d referenced", pid));
			return 0;
			}
		}

	else
		{
		DBG_LOG(DBG_SERIAL, "-- No caching");
		}

	uint16 stype;
	if ( ! UNSERIALIZE(&stype) )
		return 0;

	SerialObj* obj = Instantiate(SerialType(stype));

	if ( ! obj )
		{
		info->s->Error("unknown object type");
		return 0;
		}

	const TransientID* tid = obj->GetTID();

	if ( info->cache )
		{
		assert(tid);
		info->s->Cache()->Register(obj, pid);
		}

	info->type = stype;

#ifdef DEBUG
	obj->serial_type = stype;
#endif

	DBG_PUSH(DBG_SERIAL);
	if ( ! obj->DoUnserialize(info) )
		{
		DBG_POP(DBG_SERIAL);
		return 0;
		}

	DBG_POP(DBG_SERIAL);

	if ( SerialObj::CheckTypes(stype, type) )
		{
		info->s->Error("type mismatch");
		return 0;
		}

#ifdef DEBUG
	if ( debug_logger.IsEnabled(DBG_SERIAL) && IsBroObj(stype) )
		{
		ODesc desc(DESC_READABLE);
		((BroObj*)obj)->Describe(&desc);
		DBG_LOG(DBG_SERIAL, "-- Desc: %s", desc.Description());
		}
#endif

	assert(obj);
	return obj;
	}

bool SerialObj::DoSerialize(SerialInfo* info) const
	{
	assert(info->type != SER_NONE);

#ifdef DEBUG
	const_cast<SerialObj*>(this)->serial_type = info->type;
#endif

	DBG_LOG(DBG_SERIAL, __PRETTY_FUNCTION__);
	DBG_PUSH(DBG_SERIAL);
	bool ret = SERIALIZE(uint16(info->type));
	DBG_POP(DBG_SERIAL);
	return ret;
	}

bool SerialObj::DoUnserialize(UnserialInfo* info)
	{
	DBG_LOG(DBG_SERIAL, __PRETTY_FUNCTION__);
	return true;
	}
