// $Id: Sessions.cc,v 1.15 2005/09/02 21:18:24 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.


#include "config.h"

#include <arpa/inet.h>

#include <stdlib.h>
#include <unistd.h>

#include "Net.h"
#include "Event.h"
#include "Timer.h"
#include "NetVar.h"
#include "Sessions.h"
#include "Active.h"
#include "OSFinger.h"

#include "FTP.h"
#include "Finger.h"
#include "Gnutella.h"
#include "Ident.h"
#include "HTTP.h"
#include "SMTP.h"
#include "Portmap.h"
#include "NFS.h"
#include "Telnet.h"
#include "Rlogin.h"
#include "RSH.h"
#include "DNS.h"
#include "NetbiosSSN.h"
#include "SSH.h"
#include "NTP.h"
#include "ICMP.h"
#include "SSLProxy.h"
#include "DCE_RPC.h"
#include "SMB.h"

#include "SteppingStone.h"
#include "BackDoor.h"
#include "InterConn.h"
#include "Discard.h"
#include "RuleMatcher.h"
#include "ConnCompressor.h"

#include "PacketSort.h"


NetSessions* sessions;


class NetworkTimer : public Timer {
public:
	NetworkTimer(NetSessions* arg_sess, double arg_t)
		: Timer(arg_t, TIMER_NETWORK)
		{ sess = arg_sess; }

	void Dispatch(double t, int is_expire);

protected:
	NetSessions* sess;
};

void NetworkTimer::Dispatch(double t, int is_expire)
	{
	if ( is_expire )
		return;

	sess->HeartBeat(t);
	}


NetSessions::NetSessions()
	{
	TypeList* t = new TypeList();
	t->Append(base_type(TYPE_COUNT));	// source IP address
	t->Append(base_type(TYPE_COUNT));	// dest IP address
	t->Append(base_type(TYPE_COUNT));	// source and dest ports

	ch = new CompositeHash(t);

	Unref(t);

	tcp_conns.SetDeleteFunc(bro_obj_delete_func);
	udp_conns.SetDeleteFunc(bro_obj_delete_func);
	fragments.SetDeleteFunc(bro_obj_delete_func);

	if ( reading_live && net_stats_update )
		timer_mgr->Add(new NetworkTimer(this, 1.0));

	if ( stp_correlate_pair )
		stp_manager = new SteppingStoneManager();
	else
		stp_manager = 0;

	discarder = new Discarder();
	if ( ! discarder->IsActive() )
		{
		delete discarder;
		discarder = 0;
		}

	packet_filter = 0;

	build_backdoor_analyzer =
		backdoor_stats || rlogin_signature_found ||
		telnet_signature_found || ssh_signature_found ||
		root_backdoor_signature_found || ftp_signature_found ||
		napster_signature_found || kazaa_signature_found ||
		http_signature_found || http_proxy_signature_found;

	dump_this_packet = 0;
	num_packets_processed = 0;

	if ( OS_version_found )
		{
		SYN_OS_Fingerprinter = new OSFingerprint(SYN_FINGERPRINT_MODE);
		if ( SYN_OS_Fingerprinter->Error() )
			exit(1);
		}
	else
		SYN_OS_Fingerprinter = 0;

	if ( pkt_profile_mode && pkt_profile_freq > 0 && pkt_profile_file )
		pkt_profiler = new PacketProfiler(pkt_profile_mode,
				pkt_profile_freq, pkt_profile_file->AsFile());
	else
		pkt_profiler = 0;
	}

NetSessions::~NetSessions()
	{
	delete ch;
	delete packet_filter;
	delete pkt_profiler;
	}

void NetSessions::Done()
	{
	delete stp_manager;
	delete discarder;
	}

void NetSessions::DispatchPacket(double t, const struct pcap_pkthdr* hdr,
			const u_char* pkt, int hdr_size,
			PktSrc* src_ps)
	{
	PacketSortElement* pkt_elem = 0;

	if ( packet_sort_window > 0 )
		{
		double min_t = t - packet_sort_window;

		pkt_elem = packet_sorter->RemoveMin(min_t);

		if ( pkt_elem )
			{
			packet_sorter->Add(
				new PacketSortElement(src_ps, t, hdr, pkt, hdr_size));

			src_ps = pkt_elem->Src();
			hdr = pkt_elem->Hdr();
			pkt = pkt_elem->Pkt();
			hdr_size = pkt_elem->HdrSize();
			t = pkt_elem->TimeStamp();
			}
		}

	current_hdr = hdr;
	current_pkt = pkt;
	current_pktsrc = src_ps;

	if ( encap_hdr_size > 0 )
		{
		// We're doing tunnel encapsulation.  Check whether there's
		// a particular associated port.
		if ( tunnel_port > 0 )
			{
			struct ip* ip_hdr = (struct ip*) (pkt + hdr_size);
			if ( ip_hdr->ip_p == IPPROTO_UDP )
				{
				struct udphdr* udp_hdr = (struct udphdr*)
					(pkt + hdr_size + ip_hdr->ip_hl * 4);

				if ( ntohs(udp_hdr->uh_dport) == tunnel_port )
					// A match.
					hdr_size += encap_hdr_size;
				}
			}

		else
			// Blanket encapsulation.
			hdr_size += encap_hdr_size;
		}

	if ( src_ps->FilterType() == TYPE_FILTER_NORMAL )
		NextPacket(network_time, hdr, pkt, hdr_size, pkt_elem);
	else
		NextPacketSecondary(network_time, hdr, pkt, hdr_size, src_ps);
	}

void NetSessions::NextPacket(double t, const struct pcap_pkthdr* hdr,
			     const u_char* const pkt, int hdr_size,
			     PacketSortElement* pkt_elem)
	{
	SegmentProfiler(segment_logger, "processing-packet");
	if ( pkt_profiler )
		pkt_profiler->ProfilePkt(t, hdr->caplen);

	++num_packets_processed;

	dump_this_packet = 0;

	if ( pkt_elem && pkt_elem->IPHdr() )
		// Fast path for "normal" IP packets if an IP_Hdr is
		// already extracted when doing PacketSort. Otherwise
		// the code below tries to extract the IP header, the
		// difference here is that header extraction in
		// PacketSort does not generate Weird events.

		DoNextPacket(t, hdr, pkt_elem->IPHdr(), pkt, hdr_size);

	else
		{
		uint32 caplen = hdr->caplen - hdr_size;
		if ( caplen < sizeof(struct ip) )
			{
			Weird("truncated_IP", hdr, pkt);
			return;
			}

		const struct ip* ip = (const struct ip*) (pkt + hdr_size);
		if ( ip->ip_v == 4 )
			{
			IP_Hdr ip_hdr(ip);
			DoNextPacket(t, hdr, &ip_hdr, pkt, hdr_size);
			}
		else
			{
#ifdef BROv6
			IP_Hdr ip_hdr((const struct ip6_hdr*) (pkt + hdr_size));
			DoNextPacket(t, hdr, &ip_hdr, pkt, hdr_size);
#else
			Weird("non_IPv4_packet", hdr, pkt);
			return;
#endif
			}
		}

	if ( dump_this_packet )
		DumpPacket(hdr, pkt);
	}

void NetSessions::NextPacketSecondary(double t, const struct pcap_pkthdr* hdr,
				const u_char* const pkt, int hdr_size,
				const PktSrc* src_ps)
	{
	SegmentProfiler(segment_logger, "processing-secondary-packet");

	++num_packets_processed;

	uint32 caplen = hdr->caplen - hdr_size;
	if ( caplen < sizeof(struct ip) )
		{
		Weird("truncated_IP", hdr, pkt);
		return;
		}

	const struct ip* ip = (const struct ip*) (pkt + hdr_size);
	if ( ip->ip_v == 4 )
		{
		const secondary_program_list& spt = src_ps->ProgramTable();

		loop_over_list(spt, i)
			{
			SecondaryProgram* sp = spt[i];
			if ( ! net_packet_match(sp->Program(), pkt,
						hdr->len, hdr->caplen) )
				continue;

			val_list* args = new val_list;
			StringVal* cmd_val =
				new StringVal(sp->Event()->Filter());
			args->append(cmd_val);
			args->append(BuildHeader(ip));
			// ### Need to queue event here.
			sp->Event()->Event()->Call(args);
			delete args;
			}
		}
	}


void NetSessions::DoNextPacket(double t, const struct pcap_pkthdr* hdr,
				const IP_Hdr* ip_hdr, const u_char* const pkt,
				int hdr_size)
	{
	uint32 caplen = hdr->caplen - hdr_size;
	const struct ip* ip4 = ip_hdr->IP4_Hdr();

	uint32 len = ip_hdr->TotalLen();
	if ( hdr->len < len + hdr_size )
		{
		Weird("truncated_IP", hdr, pkt);
		return;
		}

	// Ignore if packet matches packet filter.
	if ( packet_filter && packet_filter->Match(ip_hdr, len, caplen) )
		 return;

	int ip_hdr_len = ip_hdr->HdrLen();
	if ( ! ignore_checksums && ip4 &&
	     ones_complement_checksum((void*) ip4, ip_hdr_len, 0) != 0xffff )
		{
		Weird("bad_IP_checksum", hdr, pkt);
		return;
		}

	if ( discarder && discarder->NextPacket(ip_hdr, len, caplen) )
		return;

	int proto = ip_hdr->NextProto();
	if ( proto != IPPROTO_TCP && proto != IPPROTO_UDP &&
	     proto != IPPROTO_ICMP )
		{
		dump_this_packet = 1;
		return;
		}

	// Check for TTL/MTU problems from Active Mapping
#ifndef BROv6
#ifdef ACTIVE_MAPPING
	const NumericData* numeric;
	get_map_result(ip4->ip_dst.s_addr, numeric);

	if ( numeric->hops && ip4->ip_ttl < numeric->hops )
		{
		debug_msg("Packet destined for %s had ttl %d but there are %d hops to host.\n",
		inet_ntoa(ip4->ip_dst), ip4->ip_ttl, numeric->hops);
		return;
		}
#endif
#endif

	FragReassembler* f = 0;
	uint32 frag_field = ip_hdr->FragField();

#ifndef BROv6
#ifdef ACTIVE_MAPPING
	if ( numeric->path_MTU && (frag_field & IP_DF) )
		{
		if ( htons(ip4->ip_len) > numeric->path_MTU )
			{
			debug_msg("Packet destined for %s has DF flag but its size %d is greater than pmtu of %d\n",
			inet_ntoa(ip4->ip_dst), htons(ip4->ip_len), numeric->path_MTU);
			return;
			}
		}
#endif
#endif

	if ( (frag_field & 0x3fff) != 0 )
		{
		dump_this_packet = 1;	// always record fragments

		if ( caplen < len )
			{
			Weird("incompletely_captured_fragment", ip_hdr);
			return;	// don't try to reassemble, that's doomed
			}

		f = NextFragment(t, ip_hdr, pkt + hdr_size, frag_field);
		const IP_Hdr* ih = f->ReassembledPkt();
		if ( ! ih )
			// It didn't reassemble into anything yet.
			return;

		ip4 = ih->IP4_Hdr();
		ip_hdr = ih;

		caplen = len = ip_hdr->TotalLen();
		ip_hdr_len = ip_hdr->HdrLen();
		}

	len -= ip_hdr_len;	// remove IP header
	caplen -= ip_hdr_len;

	uint32 min_hdr_len = (proto == IPPROTO_TCP) ?  sizeof(struct tcphdr) :
		(proto == IPPROTO_UDP ? sizeof(struct udphdr) : ICMP_MINLEN);

	if ( len < min_hdr_len )
		{
		Weird("truncated_header", hdr, pkt);
		if ( f )
			Remove(f);	// ###
		return;
		}
	if ( caplen < min_hdr_len )
		{
		Weird("internally_truncated_header", hdr, pkt);
		if ( f )
			Remove(f);	// ###
		return;
		}

	// Where the data starts - if this is a protocol we know about,
	// this gets advanced past the transport header.
	const u_char* data = ip_hdr->Payload();

	ConnID id;
	id.src_addr = ip_hdr->SrcAddr();
	id.dst_addr = ip_hdr->DstAddr();
	Dictionary* d = 0;
	bool pass_to_conn_compressor = false;

	switch ( proto ) {
	case IPPROTO_TCP:
		{
		const struct tcphdr* tp = (const struct tcphdr *) data;
		id.src_port = tp->th_sport;
		id.dst_port = tp->th_dport;
		id.is_one_way = 0;
		d = &tcp_conns;
		pass_to_conn_compressor = use_connection_compressor;
		break;
		}

	case IPPROTO_UDP:
		{
		const struct udphdr* up = (const struct udphdr *) data;
		id.src_port = up->uh_sport;
		id.dst_port = up->uh_dport;
		id.is_one_way = 0;

		d = &udp_conns;
		break;
		}

	case IPPROTO_ICMP:
		{
		const struct icmp* icmpp = (const struct icmp *) data;

		// Put the type and code in network order, as that's what's
		// used for regular ports.
		id.src_port = htons(icmpp->icmp_type);
		id.dst_port = htons(icmpp->icmp_code);
		id.is_one_way = 1;

		d = &icmp_conns;
		break;
		}

	default:
		Weird(fmt("unknown_protocol %d", proto), hdr, pkt);
		return;
	}

	HashKey* h = id.BuildConnKey();
	if ( ! h )
		internal_error("hash computation failed");

	Connection* conn = 0;

	if ( pass_to_conn_compressor )
		conn = conn_compressor->NextPacket(t, h, ip_hdr, hdr, pkt);

	else
		{
		conn = (Connection*) d->Lookup(h);
		if ( ! conn )
			{
			conn = NewConn(h, t, &id, data, proto);
			if ( conn )
				d->Insert(h, conn);
			}
		else
			{
			if ( conn->IsReuse(t, data) )
				{
				conn->Event(connection_reused);
				Remove(conn);
				conn = NewConn(h, t, &id, data, proto);
				if ( conn )
					d->Insert(h, conn);
				}
			else
				delete h;
			}

		if ( ! conn )
			delete h;
		}

	if ( ! conn )
		return;

	int record_packet = 1;	// whether to record the packet at all
	int record_content = 1;	// whether to record its data

	int is_orig = addr_eq(id.src_addr, conn->OrigAddr()) &&
			id.src_port == conn->OrigPort();

	if ( new_packet )
		conn->Event(new_packet, BuildHeader(ip_hdr->IP4_Hdr()));

	conn->NextPacket(t, is_orig, ip_hdr, len, caplen, data,
				record_packet, record_content,
			        hdr, pkt, hdr_size);

	int hdr_offset = data - ip_hdr->Payload();
	int contents_len = min(len, caplen) - hdr_offset;

	if ( packet_contents && contents_len > 0 )
		{
		BroString* cbs = new BroString(data, contents_len, 1);
		Val* contents = new StringVal(cbs);
		conn->Event(packet_contents, contents);
		}

	// Override content record setting according to
	// flags set by the policy script.
	if ( dump_original_packets_if_not_rewriting )
		record_packet = record_content = 1;
	if ( dump_selected_source_packets )
		record_packet = record_content = 0;

	if ( f )
		{
		// Above we already recorded the fragment in its entirety.
		f->DeleteTimer();
		Remove(f);	// ###
		}

	else if ( record_packet && ! conn->RewritingTrace() )
		{
		if ( record_content )
			dump_this_packet = 1;	// save the whole thing

		else
			{
			int hdr_len = data - pkt;
			DumpPacket(hdr, pkt, hdr_len);	// just save the header
			}
		}
	}

Val* NetSessions::BuildHeader(const struct ip* ip)
	{
	static RecordType* pkt_hdr_type = 0;
	static RecordType* ip_hdr_type = 0;
	static RecordType* tcp_hdr_type = 0;
	static RecordType* udp_hdr_type = 0;
	static RecordType* icmp_hdr_type;

	if ( ! pkt_hdr_type )
		{
		pkt_hdr_type = internal_type("pkt_hdr")->AsRecordType();
		ip_hdr_type = internal_type("ip_hdr")->AsRecordType();
		tcp_hdr_type = internal_type("tcp_hdr")->AsRecordType();
		udp_hdr_type = internal_type("udp_hdr")->AsRecordType();
		icmp_hdr_type = internal_type("icmp_hdr")->AsRecordType();
		}

	RecordVal* pkt_hdr = new RecordVal(pkt_hdr_type);

	RecordVal* ip_hdr = new RecordVal(ip_hdr_type);

	int ip_hdr_len = ip->ip_hl * 4;
	int pkt_len = ip->ip_len;

	ip_hdr->Assign(0, new Val(ip->ip_hl * 4, TYPE_COUNT));
	ip_hdr->Assign(1, new Val(ip->ip_tos, TYPE_COUNT));
	ip_hdr->Assign(2, new Val(ntohs(ip->ip_len), TYPE_COUNT));
	ip_hdr->Assign(3, new Val(ntohs(ip->ip_id), TYPE_COUNT));
	ip_hdr->Assign(4, new Val(ip->ip_ttl, TYPE_COUNT));
	ip_hdr->Assign(5, new Val(ip->ip_p, TYPE_COUNT));
	ip_hdr->Assign(6, new AddrVal(ip->ip_src.s_addr));
	ip_hdr->Assign(7, new AddrVal(ip->ip_dst.s_addr));

	pkt_hdr->Assign(0, ip_hdr);

	// L4 header.
	const u_char* data = ((const u_char*) ip) + ip_hdr_len;

	int proto = ip->ip_p;
	switch ( proto ) {
	case IPPROTO_TCP:
		{
		const struct tcphdr* tp = (const struct tcphdr*) data;
		RecordVal* tcp_hdr = new RecordVal(tcp_hdr_type);

		int tcp_hdr_len = tp->th_off * 4;
		int data_len = pkt_len - ip_hdr_len - tcp_hdr_len;

		tcp_hdr->Assign(0, new PortVal(ntohs(tp->th_sport), TRANSPORT_TCP));
		tcp_hdr->Assign(1, new PortVal(ntohs(tp->th_dport), TRANSPORT_TCP));
		tcp_hdr->Assign(2, new Val(uint32(ntohl(tp->th_seq)), TYPE_COUNT));
		tcp_hdr->Assign(3, new Val(uint32(ntohl(tp->th_ack)), TYPE_COUNT));
		tcp_hdr->Assign(4, new Val(tcp_hdr_len, TYPE_COUNT));
		tcp_hdr->Assign(5, new Val(data_len, TYPE_COUNT));
		tcp_hdr->Assign(6, new Val(tp->th_flags, TYPE_COUNT));
		tcp_hdr->Assign(7, new Val(ntohs(tp->th_win), TYPE_COUNT));

		pkt_hdr->Assign(1, tcp_hdr);
		break;
		}

	case IPPROTO_UDP:
		{
		const struct udphdr* up = (const struct udphdr*) data;
		RecordVal* udp_hdr = new RecordVal(udp_hdr_type);

		udp_hdr->Assign(0, new PortVal(ntohs(up->uh_sport), TRANSPORT_UDP));
		udp_hdr->Assign(1, new PortVal(ntohs(up->uh_dport), TRANSPORT_UDP));
		udp_hdr->Assign(2, new Val(ntohs(up->uh_ulen), TYPE_COUNT));

		pkt_hdr->Assign(2, udp_hdr);
		break;
		}

	case IPPROTO_ICMP:
		{
		const struct icmp* icmpp = (const struct icmp *) data;
		RecordVal* icmp_hdr = new RecordVal(icmp_hdr_type);

		icmp_hdr->Assign(0, new Val(icmpp->icmp_type, TYPE_COUNT));

		pkt_hdr->Assign(3, icmp_hdr);
		break;
		}

	default:
		{
		// This is not a protocol we understand.
		}
	}

	return pkt_hdr;
	}

FragReassembler* NetSessions::NextFragment(double t, const IP_Hdr* ip,
					const u_char* pkt, uint32 frag_field)
	{
	uint32 src_addr = uint32(ip->SrcAddr4());
	uint32 dst_addr = uint32(ip->DstAddr4());
	uint32 frag_id = ntohs(ip->ID4());	// we actually could skip conv.

	ListVal* key = new ListVal(TYPE_ANY);
	key->Append(new Val(src_addr, TYPE_COUNT));
	key->Append(new Val(dst_addr, TYPE_COUNT));
	key->Append(new Val(frag_id, TYPE_COUNT));

	HashKey* h = ch->ComputeHash(key, 1);
	if ( ! h )
		internal_error("hash computation failed");

	FragReassembler* f = fragments.Lookup(h);
	if ( ! f )
		{
		f = new FragReassembler(this, ip, pkt, frag_field, h, t);
		fragments.Insert(h, f);
		Unref(key);
		return f;
		}

	delete h;
	Unref(key);

	f->AddFragment(t, ip, pkt, frag_field);
	return f;
	}

int NetSessions::Get_OS_From_SYN(struct os_type* retval,
		  uint16 tot, uint8 DF_flag, uint8 TTL, uint16 WSS,
		  uint8 ocnt, uint8* op, uint16 MSS, uint8 win_scale,
		  uint32 tstamp, /* uint8 TOS, */ uint32 quirks,
		  uint8 ECN) const
	{
	return SYN_OS_Fingerprinter ?
		SYN_OS_Fingerprinter->FindMatch(retval, tot, DF_flag, TTL,
				WSS, ocnt, op, MSS, win_scale, tstamp,
				quirks, ECN) : 0;
	}

bool NetSessions::CompareWithPreviousOSMatch(uint32 addr, int id) const
	{
	return SYN_OS_Fingerprinter ?
		SYN_OS_Fingerprinter->CacheMatch(addr, id) : 0;
	}

Connection* NetSessions::FindConnection(Val* v)
	{
	if ( v->Type() != conn_id )
		return 0;

	const val_list* vl = v->AsRecord();

	addr_type orig_addr = (*vl)[0]->AsAddr();
	addr_type resp_addr = (*vl)[2]->AsAddr();

	ConnID id;

#ifdef BROv6
	id.src_addr = orig_addr;
	id.dst_addr = resp_addr;
#else
	id.src_addr = &orig_addr;
	id.dst_addr = &resp_addr;
#endif

	PortVal* orig_portv = (*vl)[1]->AsPortVal();
	PortVal* resp_portv = (*vl)[3]->AsPortVal();

	id.src_port = htons((unsigned short) orig_portv->Port());
	id.dst_port = htons((unsigned short) resp_portv->Port());

	id.is_one_way = 0;	// ### incorrect for ICMP connections

	HashKey* h = id.BuildConnKey();
	if ( ! h )
		internal_error("hash computation failed");

	Dictionary* d;

	if ( orig_portv->IsTCP() )
		{
		if ( use_connection_compressor )
			{
			Connection* conn = conn_compressor->Lookup(h);
			delete h;
			return conn;
			}
		else
			d = &tcp_conns;
		}
	else if ( orig_portv->IsUDP() )
		d = &udp_conns;
	else if ( orig_portv->IsICMP() )
		d = &icmp_conns;
	else
		internal_error("bad port value in NetSessions::FindConnection");

	Connection* conn = (Connection*) d->Lookup(h);

	delete h;

	return conn;
	}

void NetSessions::Remove(Connection* c)
	{
	HashKey* k = c->Key();
	if ( k )
		{
		c->CancelTimers();
		c->Done();

		if ( c->ConnTransport() == TRANSPORT_TCP )
			{
			TCP_Connection* tc = (TCP_Connection*) c;
			TCP_Endpoint* to = tc->Orig();
			TCP_Endpoint* tr = tc->Resp();

			tcp_stats.StateLeft(to->state, tr->state);
			}

		if ( connection_state_remove )
			c->Event(connection_state_remove);

		if ( c->IsPersistent() )
			persistence_serializer->Unregister(c);

		// Zero out c's copy of the key, so that if c has been Ref()'d
		// up, we know on a future call to Remove() that it's no
		// longer in the dictionary.
		c->ClearKey();

		switch ( c->ConnTransport() ) {
		case TRANSPORT_TCP:
			if ( use_connection_compressor )
				conn_compressor->Remove(k);
			else if ( ! tcp_conns.RemoveEntry(k) )
				internal_error(fmt("connection missing"));

			((TCP_Connection*) c)->AnalyzerDone();
			break;

		case TRANSPORT_UDP:
			if ( ! udp_conns.RemoveEntry(k) )
				internal_error("connection missing");
			break;

		case TRANSPORT_ICMP:
			if ( ! icmp_conns.RemoveEntry(k) )
				internal_error("connection missing");
			break;
		}

		Unref(c);
		delete k;
		}
	}

void NetSessions::Remove(FragReassembler* f)
	{
	HashKey* k = f->Key();
	if ( ! k )
		internal_error("fragment block not in dictionary");

	if ( ! fragments.RemoveEntry(k) )
		internal_error("fragment block missing");

	Unref(f);
	}

void NetSessions::Insert(Connection* c)
	{
	assert(c->Key());

	Connection* old = 0;

	switch ( c->ConnTransport() ) {
	// Remove first. Otherwise the dictioanry would still
	// reference the old key for already existing connections.

	case TRANSPORT_TCP:
		old = (Connection*) tcp_conns.Remove(c->Key());
		tcp_conns.Insert(c->Key(), (TCP_Connection*) c);
		break;

	case TRANSPORT_UDP:
		old = (Connection*) udp_conns.Remove(c->Key());
		udp_conns.Insert(c->Key(), (UDP_Connection*) c);
		break;

	case TRANSPORT_ICMP:
		old = (Connection*) icmp_conns.Remove(c->Key());
		icmp_conns.Insert(c->Key(), (ICMP_Connection*) c);
		break;

	default:
		internal_error("unknown connection type");
	}

	if ( old && old != c )
		{
		// Some clean-ups similar to those in Remove() (but invisible
		// to the script layer).
		old->CancelTimers();
		if ( old->IsPersistent() )
			persistence_serializer->Unregister(old);
		delete old->Key();
		old->ClearKey();
		Unref(old);
		}
	}

void NetSessions::Drain()
	{
	if ( use_connection_compressor )
		conn_compressor->Drain();

	IterCookie* cookie = tcp_conns.InitForIteration();
	TCP_Connection* tc;

	while ( (tc = tcp_conns.NextEntry(cookie)) )
		{
		tc->Done();
		tc->Event(connection_state_remove);
		}

	cookie = udp_conns.InitForIteration();
	UDP_Connection* uc;

	while ( (uc = udp_conns.NextEntry(cookie)) )
		{
		uc->Done();
		uc->Event(connection_state_remove);
		}

	cookie = icmp_conns.InitForIteration();
	ICMP_Connection* ic;

	while ( (ic = icmp_conns.NextEntry(cookie)) )
		{
		ic->Done();
		ic->Event(connection_state_remove);
		}
	}

void NetSessions::HeartBeat(double t)
	{
	unsigned int recv = 0;
	unsigned int drop = 0;
	unsigned int link = 0;

	loop_over_list(pkt_srcs, i)
		{
		PktSrc* ps = pkt_srcs[i];

		struct PktSrc::Stats stat;
		ps->Statistics(&stat);
		recv += stat.received;
		drop += stat.dropped;
		link += stat.link;
		}

	val_list* vl = new val_list;

	vl->append(new Val(t, TYPE_TIME));

	RecordVal* ns = new RecordVal(net_stats);
	ns->Assign(0, new Val(recv, TYPE_COUNT));
	ns->Assign(1, new Val(drop, TYPE_COUNT));
	ns->Assign(2, new Val(link, TYPE_COUNT));

	vl->append(ns);

	mgr.QueueEvent(net_stats_update, vl);

	timer_mgr->Add(new NetworkTimer(this, t + heartbeat_interval));
	}

void NetSessions::GetStats(SessionStats& s) const
	{
	s.num_TCP_conns = tcp_conns.Length();
	s.num_UDP_conns = udp_conns.Length();
	s.num_ICMP_conns = icmp_conns.Length();
	s.num_fragments = fragments.Length();
	s.num_packets = num_packets_processed;
	s.num_timers = timer_mgr->Size();
	s.num_events_queued = num_events_queued;
	s.num_events_dispatched = num_events_dispatched;

	s.max_TCP_conns = tcp_conns.MaxLength();
	s.max_UDP_conns = udp_conns.MaxLength();
	s.max_ICMP_conns = icmp_conns.MaxLength();
	s.max_fragments = fragments.MaxLength();
	s.max_timers = timer_mgr->PeakSize();
	}

Connection* NetSessions::NewConn(HashKey* k, double t, const ConnID* id,
		const u_char* data, int proto)
	{
	switch ( proto ) {
	case IPPROTO_TCP:
		return NewConn(k, t, id, (const struct tcphdr *) data);

	case IPPROTO_UDP:
		return NewConn(k, t, id, (const struct udphdr *) data);

	case IPPROTO_ICMP:
		return NewConn(k, t, id, (const struct icmp *) data);

	default:
		internal_error("bad protocol in NetSessions::NewConn");
		return 0;
	}
	}

TCP_Connection* NetSessions::NewConn(HashKey* k, double t, const ConnID* orig_id,
			const struct tcphdr* tp)
	{
	TCP_Connection* c = 0;
	const ConnID* id = orig_id;
	ConnID flip_id;	// only used for flip_roles below

	bool flip_roles;
	if ( ! WantTCPConnection(orig_id->src_port, orig_id->dst_port,
			tp->th_flags, flip_roles) )
		return 0;

	if ( flip_roles )
		{
		flip_id = *id;

		const uint32* ta = flip_id.src_addr;
		flip_id.src_addr = flip_id.dst_addr;
		flip_id.dst_addr = ta;

		uint32 t = flip_id.src_port;
		flip_id.src_port = flip_id.dst_port;
		flip_id.dst_port = t;

		id = &flip_id;
		}

	int is_analyzed = 1;
	int service = ntohs(id->dst_port);
	switch ( service ) {
	// NOTE: Reflect changes here in IsLikelyServerPort().
	case 21:
		if ( ftp_request || ftp_reply )
			c = new FTP_Conn(this, k, t, id, tp);
		break;

	case 22:
		if ( ssh_client_version || ssh_server_version )
			c = new SSH_Conn(this, k, t, id, tp);
		break;

	case 23:
		if ( login_failure || login_success ||
		     login_input_line || login_output_line )
			c = new TelnetConn(this, k, t, id, tp);
		break;

	case 25:
		if ( smtp_request || smtp_reply || smtp_data || smtp_unexpected )
			c = new SMTP_Conn(this, k, t, id, tp);
		break;

	case 513:
		if ( login_failure || login_success ||
		     login_input_line || login_output_line )
			c = new RloginConn(this, k, t, id, tp);
		break;

	case 514:
	        if ( rsh_request || rsh_reply )
	                c = new RshConn(this, k, t, id, tp);
		break;

	case 79:
		if ( finger_request || finger_reply )
			c = new FingerConn(this, k, t, id, tp);
		break;

	case 113:
		if ( ident_request || ident_reply || ident_error )
			c = new IdentConn(this, k, t, id, tp);
		break;

	case 80:
	case 8080:
	case 8000:
	case 3128:	// Default port of Squid Proxy Cache.
		if ( http_request || http_reply )
			c = new HTTP_Conn(this, k, t, id, tp);
		break;

	case 53:
		if ( dns_request || dns_full_request )
			c = new TCP_DNS(this, k, t, id, tp);
		break;

	case 111:
		if ( pm_request )
			c = new PortmapperConn(this, k, t, id, tp);
		break;

	case 2049:
		if ( nfs_request_getattr )
			c = new NFS_Conn(this, k, t, id, tp);
		break;

	case 135:
	case 1025:
		if ( dce_rpc_request || dce_rpc_reply )
			c = new DCE_RPC_Conn(this, k, t, id, tp);
		break;

	case 139:
		{
		bool smb_events =
			smb_message || smb_com_tree_connect_andx ||
			smb_com_nt_create_andx || smb_com_transaction ||
			smb_com_transaction2 || smb_com_read_andx ||
			smb_com_write_andx;

		if ( netbios_session_request || netbios_session_accepted ||
		     netbios_session_rejected ||
		     smb_events || dce_rpc_request || dce_rpc_reply )
			c = new TCP_NetbiosSSN(this, k, t, id, tp);
		}
		break;

	case 445:
		{
		bool smb_events =
			smb_message || smb_com_tree_connect_andx ||
			smb_com_nt_create_andx || smb_com_transaction ||
			smb_com_transaction2 || smb_com_read_andx ||
			smb_com_write_andx;

		if ( smb_events || dce_rpc_request || dce_rpc_reply )
			c = new SMB_TCP_Conn(this, k, t, id, tp);
		}
		break;

#ifdef USE_OPENSSL
	case 443:	// https
	case 563:	// nntps
	case 585:	// imap4-ssl (old, use imaps)
	case 614:	// sshell
	case 636:	// ldaps
	case 989:	// ftps-data
	case 990:	// ftps
	case 992:	// telnets
	case 993:	// imaps
	case 994:	// ircs
	case 995:	// pop3s
		if ( ssl_certificate_seen || ssl_certificate ||
		     ssl_conn_attempt || ssl_conn_server_reply ||
		     ssl_conn_established || ssl_conn_reused || ssl_conn_alert)
			c = new SSL_ConnectionProxy(this, k, t, id, tp);
		break;
#endif

	case 6346:
	case 8436:
		if ( gnutella_text_msg || gnutella_binary_msg ||
		     gnutella_partial_binary_msg || gnutella_establish ||
		     gnutella_not_establish || gnutella_http_notify )
		    c = new GnutellaConn (this, k, t, id, tp);
		break;
	}

	if ( ! c )
		{
		int reass = TCP_CONTENTS_NONE;

		if ( new_connection_contents )
			reass = TCP_CONTENTS_BOTH;

		if ( rule_matcher )
			{
			Val dst_port_val(ntohs(id->dst_port), TYPE_PORT);

			if ( tcp_reassembler_ports_orig->Lookup(&dst_port_val) )
				reass |= TCP_CONTENTS_ORIG;

			if ( tcp_reassembler_ports_resp->Lookup(&dst_port_val) )
				reass |= TCP_CONTENTS_RESP;
			}

		if ( tcp_contents )
			{
			Val dst_port_val(ntohs(id->dst_port), TYPE_PORT);
			Val* result;

			result = tcp_content_delivery_ports_orig->Lookup(&dst_port_val);
			if ( result && result->AsBool() )
				reass |= TCP_CONTENTS_ORIG;

			result = tcp_content_delivery_ports_resp->Lookup(&dst_port_val);
			if ( result && result->AsBool() )
				reass |= TCP_CONTENTS_RESP;
			}

		if ( reass != TCP_CONTENTS_NONE )
			c = new TCP_ConnectionContents(this, k, t, id, tp,
						TCPContentsType(reass));
		else
			{
			c = new TCP_Connection(this, k, t, id, tp);
			is_analyzed = (non_analyzed_lifetime == 0.0);
			}
		}

	c->Init();

	if ( conn_stats )
		c->AddAnalyzer(new TCP_Stats(c));

	if ( stp_correlate_pair &&
	     (service == 22 || service == 23 || service == 513) )
		c->AddAnalyzer(new SteppingStoneAnalyzer(c, stp_manager));

	if ( build_backdoor_analyzer )
		c->AddAnalyzer(new BackDoorAnalyzer(c));

	if ( interconn_stats )
		c->AddAnalyzer(new InterConnAnalyzer(c));

	if ( new_connection )
		c->Event(new_connection);

	else if ( new_connection_contents )
		c->Event(new_connection_contents);

	if ( ! is_analyzed && ! c->HasAnalyzers() )
		c->SetLifetime(non_analyzed_lifetime);

	return c;
	}

int NetSessions::IsLikelyServerPort(uint32 port) const
	{
	switch ( port ) {	// reflect changes here above
	case 21:
	case 22:
	case 23:
	case 25:
	case 513:
	case 79:
	case 113:
	case 80:
	case 8080:
	case 8000:
	case 3128:
	case 53:
	case 111:
	case 139:
	case 6346:
	case 8436:

#ifdef USE_OPENSSL
	// SSL-relatd ports:
	case 443:
	case 563:
	case 585:
	case 614:
	case 636:
	case 989:
	case 990:
	case 992:
	case 993:
	case 994:
	case 995:
#endif

	// Not analyzed (yet), but give a hint which side the server is.
	case 110:
	case 143:
		return 1;

	default:
		return 0;
	}
	}

UDP_Connection* NetSessions::NewConn(HashKey* k, double t, const ConnID* orig_id,
			const struct udphdr* up)
	{
	UDP_Connection* c = 0;
	const ConnID* id = orig_id;
	ConnID flip_id;	// only used for flip_roles below

	int src_h = ntohs(id->src_port);
	int dst_h = ntohs(id->dst_port);

	if ( (src_h == 53 || src_h == 111 || src_h == 123) &&
	     (dst_h != 53 && dst_h != 111 && dst_h != 123) )
		{
		// Make a guess that we're seeing the tail half of
		// an analyzable connection.
		flip_id = *id;

		const uint32* ta = flip_id.src_addr;
		flip_id.src_addr = flip_id.dst_addr;
		flip_id.dst_addr = ta;

		uint32 t = flip_id.src_port;
		flip_id.src_port = flip_id.dst_port;
		flip_id.dst_port = t;

		id = &flip_id;
		}

	switch ( ntohs(id->dst_port) ) {
	case 53:
		if ( dns_request || dns_full_request )
			c = new UDP_DNS(this, k, t, id, up);
		break;

	case 111:
		if ( pm_request )
			c = new UDP_Portmapper(this, k, t, id, up);
		break;

	case 123:
		if ( ntp_message )
			c = new NTP_Session(this, k, t, id, up);
		break;

	case 2049:
		if ( nfs_request_getattr )
			c = new UDP_NFS(this, k, t, id, up);
		break;
	}

	if ( ! c )
		{
		c = new UDP_Connection(this, k, t, id, up);

		if ( non_analyzed_lifetime != 0.0 )
			c->SetLifetime(non_analyzed_lifetime);
		}

	if ( new_connection )
		c->Event(new_connection);

	return c;
	}

ICMP_Connection* NetSessions::NewConn(HashKey* k, double t, const ConnID* id,
			const struct icmp* icmpp)
	{
	switch ( icmpp->icmp_type ) {
	case ICMP_ECHO:
	case ICMP_ECHOREPLY:
		if ( icmp_echo_request || icmp_echo_reply )
			return new ICMP_Echo(this, k, t, id, icmpp);

	case ICMP_UNREACH:
		if ( icmp_unreachable )
			return new ICMP_Context(this, k, t, id, icmpp);

	case ICMP_TIMXCEED:
		if ( icmp_time_exceeded )
			return new ICMP_Context(this, k, t, id, icmpp);

	default:
		ICMP_Connection* c = new ICMP_Connection(this, k, t, id, icmpp);

		if ( non_analyzed_lifetime != 0.0 )
			c->SetLifetime(non_analyzed_lifetime);

		return c;
	}
	}

bool NetSessions::WantTCPConnection(uint16 src_port, uint16 dst_port,
		uint8 tcp_flags, bool& flip_roles)
	{
	flip_roles = false;

	if ( ! (tcp_flags & TH_SYN) || (tcp_flags & TH_ACK) )
		{
		// The new connection is starting either without a SYN,
		// or with a SYN ack. This means it's a partial connection.
		if ( ! partial_connection_ok )
			return false;

		if ( tcp_flags & TH_SYN && ! tcp_SYN_ack_ok )
			return false;

		// Try to guess true responder by the port numbers.
		// (We might also think that for SYN acks we could
		// safely flip the roles, but that doesn't work
		// for stealth scans.)
		flip_roles =
			IsLikelyServerPort(ntohs(src_port)) &&
			! IsLikelyServerPort(ntohs(dst_port));
		}

	return true;
	}

void NetSessions::DumpPacket(const struct pcap_pkthdr* hdr,
				const u_char* pkt, int len)
	{
	if ( ! pkt_dumper )
		return;

	if ( len == 0 )
		pkt_dumper->Dump(hdr, pkt);
	else
		{
		struct pcap_pkthdr h = *hdr;
		h.caplen = len;
		if ( h.caplen > hdr->caplen )
			internal_error("bad modified caplen");
		pkt_dumper->Dump(&h, pkt);
		}
	}

void NetSessions::Internal(const char* msg, const struct pcap_pkthdr* hdr,
				const u_char* pkt)
	{
	DumpPacket(hdr, pkt);
	internal_error(msg);
	}

void NetSessions::Weird(const char* name,
			const struct pcap_pkthdr* hdr, const u_char* pkt)
	{
	if ( hdr )
		dump_this_packet = 1;

	if ( net_weird )
		{
		val_list* vl = new val_list;
		vl->append(new StringVal(name));
		mgr.QueueEvent(net_weird, vl);
		}
	else
		fprintf(stderr, "weird: %.06f %s\n", network_time, name);
	}

void NetSessions::Weird(const char* name, const IP_Hdr* ip)
	{
	if ( flow_weird )
		{
		val_list* vl = new val_list;
		vl->append(new StringVal(name));
		vl->append(new AddrVal(ip->SrcAddr4()));
		vl->append(new AddrVal(ip->DstAddr4()));
		mgr.QueueEvent(flow_weird, vl);
		}
	else
		fprintf(stderr, "weird: %.06f %s\n", network_time, name);
	}

unsigned int NetSessions::ConnectionMemoryUsage()
	{
	unsigned int mem = 0;

	IterCookie* cookie = tcp_conns.InitForIteration();
	TCP_Connection* tc;

	while ( (tc = tcp_conns.NextEntry(cookie)) )
		mem += tc->MemoryAllocation();

	cookie = udp_conns.InitForIteration();
	UDP_Connection* uc;

	while ( (uc = udp_conns.NextEntry(cookie)) )
		mem += uc->MemoryAllocation();

	cookie = icmp_conns.InitForIteration();
	ICMP_Connection* ic;

	while ( (ic = icmp_conns.NextEntry(cookie)) )
		mem += ic->MemoryAllocation();

	return mem;
	}

unsigned int NetSessions::ConnectionMemoryUsageConnVals()
	{
	unsigned int mem = 0;

	IterCookie* cookie = tcp_conns.InitForIteration();
	TCP_Connection* tc;

	while ( (tc = tcp_conns.NextEntry(cookie)) )
		mem += tc->MemoryAllocationConnVal();

	cookie = udp_conns.InitForIteration();
	UDP_Connection* uc;

	while ( (uc = udp_conns.NextEntry(cookie)) )
		mem += uc->MemoryAllocationConnVal();

	cookie = icmp_conns.InitForIteration();
	ICMP_Connection* ic;

	while ( (ic = icmp_conns.NextEntry(cookie)) )
		mem += ic->MemoryAllocationConnVal();

	return mem;
	}

unsigned int NetSessions::MemoryAllocation()
	{
	return ConnectionMemoryUsage()
		+ padded_sizeof(*this)
		+ ch->MemoryAllocation()
		// must take care we don't count the HaskKeys twice.
		+ tcp_conns.MemoryAllocation() - padded_sizeof(tcp_conns) -
		// 12 is sizeof(Key) from ConnID::BuildConnKey();
		// it can't be (easily) accessed here. :-(
			(tcp_conns.Length() * pad_size(12))
		+ udp_conns.MemoryAllocation() - padded_sizeof(udp_conns) -
			(udp_conns.Length() * pad_size(12))
		+ icmp_conns.MemoryAllocation() - padded_sizeof(icmp_conns) -
			(icmp_conns.Length() * pad_size(12))
		+ fragments.MemoryAllocation() - padded_sizeof(fragments)
		// FIXME: MemoryAllocation() not implemented for rest.
		;
	}
