// $Id: StateAccess.cc,v 1.3 2004/11/02 06:57:17 vern Exp $

#include "Val.h"
#include "StateAccess.h"
#include "Serializer.h"
#include "Event.h"
#include "NetVar.h"
#include "DebugLogger.h"
#include "RemoteSerializer.h"

bool StateAccess::replaying = false;

StateAccess::StateAccess(Opcode arg_opcode,
		const MutableVal* arg_target, const Val* arg_op1,
		const Val* arg_op2, const Val* arg_op3)
	{
	opcode = arg_opcode;
	target.val = const_cast<MutableVal*>(arg_target);
	target_type = TYPE_MVAL;
	op1.val = const_cast<Val*>(arg_op1);
	op1_type = TYPE_VAL;
	op2 = const_cast<Val*>(arg_op2);
	op3 = const_cast<Val*>(arg_op3);

	RefThem();
	}

StateAccess::StateAccess(Opcode arg_opcode,
		const ID* arg_target, const Val* arg_op1,
		const Val* arg_op2, const Val* arg_op3)
	{
	opcode = arg_opcode;
	target.id = const_cast<ID*>(arg_target);
	target_type = TYPE_ID;
	op1.val = const_cast<Val*>(arg_op1);
	op1_type = TYPE_VAL;
	op2 = const_cast<Val*>(arg_op2);
	op3 = const_cast<Val*>(arg_op3);

	RefThem();
	}

StateAccess::StateAccess(Opcode arg_opcode,
		const ID* arg_target, const HashKey* arg_op1,
		const Val* arg_op2, const Val* arg_op3)
	{
	opcode = arg_opcode;
	target.id = const_cast<ID*>(arg_target);
	target_type = TYPE_ID;
	op1.key = arg_op1;
	op1_type = TYPE_KEY;
	op2 = const_cast<Val*>(arg_op2);
	op3 = const_cast<Val*>(arg_op3);
	RefThem();
	}

StateAccess::StateAccess(Opcode arg_opcode,
		const MutableVal* arg_target, const HashKey* arg_op1,
		const Val* arg_op2, const Val* arg_op3)
	{
	opcode = arg_opcode;
	target.val = const_cast<MutableVal*>(arg_target);
	target_type = TYPE_MVAL;
	op1.key = arg_op1;
	op1_type = TYPE_KEY;
	op2 = const_cast<Val*>(arg_op2);
	op3 = const_cast<Val*>(arg_op3);

	RefThem();
	}

StateAccess::~StateAccess()
	{
	if ( target_type == TYPE_ID )
		Unref(target.id);
	else
		Unref(target.val);

	if ( op1_type == TYPE_VAL )
		Unref(op1.val);

	Unref(op2);
	Unref(op3);
	}

void StateAccess::RefThem()
	{
	if ( target_type == TYPE_ID )
		Ref(target.id);
	else
		Ref(target.val);

	if ( op1_type == TYPE_VAL && op1.val )
		Ref(op1.val);

	if ( op2 )
		Ref(op2);
	if ( op3 )
		Ref(op3);
	}

bool StateAccess::CheckOld(const char* op, ID* id, Val* index,
				Val* should, Val* is)
	{
	if ( ! should && ! is )
		return true;

	// 'should == index' means that 'is' should be non-nil.
	if ( should == index && is )
		return true;

	if ( should && is )
		{
		// There's no general comparision for non-atomic vals currently.
		if ( ! (is_atomic_val(is) && is_atomic_val(should)) )
			return true;

		if ( same_atomic_val(should, is) )
			return true;
		}

	Val* arg1;
	Val* arg2;
	Val* arg3;

	if ( index )
		{
		ODesc d;
		d.SetShort();
		index->Describe(&d);
		arg1 = new StringVal(fmt("%s[%s]", id->Name(), d.Description()));
		}
	else
		arg1 = new StringVal(id->Name());

	if ( should )
		{
		ODesc d;
		d.SetShort();
		should->Describe(&d);
		arg2 = new StringVal(d.Description());
		}
	else
		arg2 = new StringVal("<none>");

	if ( is )
		{
		ODesc d;
		d.SetShort();
		is->Describe(&d);
		arg3 = new StringVal(d.Description());
		}
	else
		arg3 = new StringVal("<none>");

	val_list* args = new val_list;
	args->append(new StringVal(op));
	args->append(arg1);
	args->append(arg2);
	args->append(arg3);
	mgr.QueueEvent(remote_state_inconsistency, args);

	return false;
	}

bool StateAccess::CheckOldSet(const char* op, ID* id, Val* index,
				bool should, bool is)
	{
	if ( should == is )
		return true;

	ODesc d;
	d.SetShort();
	index->Describe(&d);

	Val* arg1 = new StringVal(fmt("%s[%s]", id->Name(), d.Description()));
	Val* arg2 = new StringVal(should ? "set" : "not set");
	Val* arg3 = new StringVal(is ? "set" : "not set");

	val_list* args = new val_list;
	args->append(new StringVal(op));
	args->append(arg1);
	args->append(arg2);
	args->append(arg3);
	mgr.QueueEvent(remote_state_inconsistency, args);

	return false;
	}

void StateAccess::Replay()
	{
	// For simplicity we assume that we only replay unserialized accesses.
	assert(target_type == TYPE_ID && op1_type == TYPE_VAL);

	if ( ! target.id )
		return;

	Val* v = target.id->ID_Val();

	if ( ! v )
		return;

	TypeTag t = v->Type()->Tag();

	replaying = true;

	switch ( opcode ) {
	case OP_ASSIGN:
		assert(op1.val);
		// There mustn't be a direct assignment to a unique ID.
		assert(target.id->Name()[0] != '#');
		CheckOld("assign", target.id, 0, op1.val, v);
		target.id->SetVal(op1.val->Ref());
		break;

	case OP_INCR:
		assert(op1.val && op2);
		if ( IsIntegral(t) )
			{
			// We derive the amount as difference between old
			// and new value.
			int amount =
				op1.val->CoerceToInt() - op2->CoerceToInt();

			target.id->SetVal(new Val(v->CoerceToInt() + amount, t),
						OP_INCR);
			}
		break;

	case OP_ASSIGN_IDX:
		assert(op1.val);

		if ( t == TYPE_TABLE )
			{
			assert(op2);
			CheckOld("index assign", target.id, op1.val, op3,
					v->AsTableVal()->Lookup(op1.val));
			v->AsTableVal()->Assign(op1.val, op2 ? op2->Ref() : 0);
			}

		else if ( t == TYPE_RECORD )
			{
			const char* field = op1.val->AsString()->CheckString();
			int idx = v->Type()->AsRecordType()->FieldOffset(field);
			if ( idx >= 0 )
				{
				CheckOld("index assign", target.id, op1.val, op3,
					v->AsRecordVal()->Lookup(idx));
				v->AsRecordVal()->Assign(idx, op2 ? op2->Ref() : 0);
				}
			else
				run_time(fmt("access replay: unknown record field %s for assign", field));
			}
		break;

	case OP_INCR_IDX:
		{
		assert(op1.val && op2 && op3);

		// We derive the amount as the difference between old
		// and new value.
		int amount = op2->CoerceToInt() - op3->CoerceToInt();

		if ( t == TYPE_TABLE )
			{
			t = v->Type()->AsTableType()->YieldType()->Tag();
			Val* lookup_op1 = v->AsTableVal()->Lookup(op1.val);
			int delta = lookup_op1->CoerceToInt() + amount;
			Val* new_val = new Val(delta, t);
			v->AsTableVal()->Assign(op1.val, new_val, OP_INCR );
			}

		else if ( t == TYPE_RECORD )
			{
			const char* field = op1.val->AsString()->CheckString();
			int idx = v->Type()->AsRecordType()->FieldOffset(field);
			if ( idx >= 0 )
				{
				t = v->Type()->AsRecordType()->FieldType(idx)->Tag();
				Val* lookup_field =
					v->AsRecordVal()->Lookup(idx);
				int delta =
					lookup_field->CoerceToInt() + amount;
				Val* new_val = new Val(delta, t);
				v->AsRecordVal()->Assign(idx, new_val, OP_INCR);
				}
			else
				run_time(fmt("access replay: unknown record field %s for assign", field));
			}
		}
		break;

	case OP_ADD:
		assert(op1.val);
		if ( t == TYPE_TABLE )
			{
			CheckOldSet("add", target.id, op1.val, op2 != 0,
					v->AsTableVal()->Lookup(op1.val) != 0);
			v->AsTableVal()->Assign(op1.val, 0);
			}
		break;

	case OP_DEL:
		assert(op1.val);
		if ( t == TYPE_TABLE )
			{
			if ( v->Type()->AsTableType()->IsSet() )
				CheckOldSet("delete", target.id, op1.val, op2 != 0,
					v->AsTableVal()->Lookup(op1.val) != 0);
			else
				CheckOld("delete", target.id, op1.val, op2,
					v->AsTableVal()->Lookup(op1.val));

			v->AsTableVal()->Delete(op1.val);
			}
		break;

	case OP_EXPIRE:
		assert(op1.val);
		if ( t == TYPE_TABLE )
			// No old check for expire. It may have already
			// been deleted by ourselves.
			v->AsTableVal()->Delete(op1.val);
		break;

	case OP_PRINT:
		assert(op1.val);
		internal_error("access replay for print not implemented");
		break;

	default:
		internal_error("access replay: unknown opcode for StateAccess");
		break;
		}

	replaying = false;

	}

bool StateAccess::Serialize(SerialInfo* info) const
	{
	return SerialObj::Serialize(info);
	}

StateAccess* StateAccess::Unserialize(UnserialInfo* info)
	{
	return (StateAccess*) SerialObj::Unserialize(info, SER_STATE_ACCESS);
	}

IMPLEMENT_SERIAL(StateAccess, SER_STATE_ACCESS);

bool StateAccess::DoSerialize(SerialInfo* info) const
	{
	DO_SERIALIZE(SER_STATE_ACCESS, SerialObj);

	if ( ! SERIALIZE(char(opcode)) )
		return false;

	const ID* id =
		target_type == TYPE_ID ? target.id : target.val->UniqueID();

	if ( ! SERIALIZE(id->Name()) )
		 return false;

	if ( op1_type == TYPE_KEY )
		{
		Val* index =
			id->ID_Val()->AsTableVal()->RecoverIndex(this->op1.key);

		if ( ! index )
			return false;
		if ( ! index->Serialize(info) )
			return false;

		Unref(index);
		}

	else if ( ! op1.val->Serialize(info) )
		return false;

	SERIALIZE_OPTIONAL(op2);
	SERIALIZE_OPTIONAL(op3);

	return true;
	}

bool StateAccess::DoUnserialize(UnserialInfo* info)
	{
	DO_UNSERIALIZE(SerialObj);

	char c;
	if ( ! UNSERIALIZE(&c) )
		return false;

	opcode = Opcode(c);

	const char* name;
	if ( ! UNSERIALIZE_STR(&name, 0) )
		return false;

	target_type = TYPE_ID;
	target.id = global_scope()->Lookup(name);

	delete [] name;

	op1_type = TYPE_VAL;
	op1.val = Val::Unserialize(info);
	if ( ! op1.val )
		return false;

	UNSERIALIZE_OPTIONAL(op2, Val::Unserialize(info));
	UNSERIALIZE_OPTIONAL(op3, Val::Unserialize(info));

	if ( target.id )
		Ref(target.id);
	else
		{
		// This may happen as long as we haven't agreed on the
		// unique name for an ID during initial synchronization.
		DBG_LOG(DBG_STATE, "state access referenced unknown id %s", name);
		}

	return true;
	}

void StateAccess::Describe(ODesc* d) const
	{
	const ID* id;
	const char* id_str = "";
	const char* unique_str = "";

	d->SetShort();

	if ( target_type == TYPE_ID )
		{
		id = target.id;

		if ( ! id )
			{
			d->Add("(unknown id)");
			return;
			}

		id_str = id->Name();

		if ( id->ID_Val() && id->ID_Val()->IsMutableVal() &&
		     id->Name()[0] != '#' )
			unique_str = fmt(" (%s)", id->ID_Val()->AsMutableVal()->UniqueID()->Name());
		}
	else
		{
		id = target.val->UniqueID();

#ifdef DEBUG
		if ( target.val->GetID() )
			{
			id_str = target.val->GetID()->Name();
			unique_str = fmt(" (%s)", id->Name());
			}
		else
#endif
			id_str = id->Name();
		}

	const Val* op1 = op1_type == TYPE_VAL ?
		this->op1.val :
		id->ID_Val()->AsTableVal()->RecoverIndex(this->op1.key);

	switch ( opcode ) {
	case OP_ASSIGN:
		assert(op1);
		d->Add(id_str);
		d->Add(" = ");
		op1->Describe(d);
		if ( op2 )
			{
			d->Add(" (");
			op2->Describe(d);
			d->Add(")");
			}
		d->Add(unique_str);
		break;

	case OP_INCR:
		assert(op1 && op2);
		d->Add(id_str);
		d->Add(" += ");
		d->Add(op1->CoerceToInt() - op2->CoerceToInt());
		d->Add(unique_str);
		break;

	case OP_ASSIGN_IDX:
		assert(op1);
		d->Add(id_str);
		d->Add("[");
		op1->Describe(d);
		d->Add("]");
		d->Add(" = ");
		if ( op2 )
			op2->Describe(d);
		else
			d->Add("(null)");
		if ( op3 )
			{
			d->Add(" (");
			op3->Describe(d);
			d->Add(")");
			}
		d->Add(unique_str);
		break;

	case OP_INCR_IDX:
		assert(op1 && op2 && op3);
		d->Add(id_str);
		d->Add("[");
		op1->Describe(d);
		d->Add("]");
		d->Add(" += ");
		d->Add(op2->CoerceToInt() - op3->CoerceToInt());
		d->Add(unique_str);
		break;

	case OP_ADD:
		assert(op1);
		d->Add("add ");
		d->Add(id_str);
		d->Add("[");
		op1->Describe(d);
		d->Add("]");
		if ( op2 )
			{
			d->Add(" (");
			op2->Describe(d);
			d->Add(")");
			}
		d->Add(unique_str);
		break;

	case OP_DEL:
		assert(op1);
		d->Add("del ");
		d->Add(id_str);
		d->Add("[");
		op1->Describe(d);
		d->Add("]");
		if ( op2 )
			{
			d->Add(" (");
			op2->Describe(d);
			d->Add(")");
			}
		d->Add(unique_str);
		break;

	case OP_EXPIRE:
		assert(op1);
		d->Add("expire ");
		d->Add(id_str);
		d->Add("[");
		op1->Describe(d);
		d->Add("]");
		if ( op2 )
			{
			d->Add(" (");
			op2->Describe(d);
			d->Add(")");
			}
		d->Add(unique_str);
		break;

	case OP_PRINT:
		assert(op1);
		d->Add("print ");
		d->Add(id_str);
		op1->Describe(d);
		d->Add(unique_str);
		break;

	default:
		internal_error("unknown opcode for StateAccess");
		break;
		}

	if ( op1_type != TYPE_VAL )
		Unref(const_cast<Val*>(op1));
	}

void StateAccess::Log(StateAccess* access)
	{
	bool synchronized = false;
	bool persistent = false;

	if ( access->target_type == TYPE_ID )
		{
		if ( access->target.id->FindAttr(ATTR_SYNCHRONIZED) )
			synchronized = true;

		if ( access->target.id->FindAttr(ATTR_PERSISTENT) )
			persistent = true;
		}
	else
		{
		if ( access->target.val->GetProperties() & MutableVal::SYNCHRONIZED )
			synchronized = true;

		if ( access->target.val->GetProperties() & MutableVal::PERSISTENT )
			persistent = true;
		}

#if 0
	// Unfortunately, this doesn't work: when rereading a
	// state file the referenced IDs are unknown. We would have
	// to write an initial assign to all persistent/synchronized
	// IDs into the file.
	if ( event_serializer && (synchronized || persistent) )
		{
		SerialInfo info(event_serializer);
		event_serializer->Serialize(&info, *access);
		}
#endif
	if ( synchronized )
		{
		SerialInfo info(remote_serializer);
		remote_serializer->SendAccess(&info, *access);
		}

#ifdef DEBUG
	ODesc desc;
	access->Describe(&desc);
	DBG_LOG(DBG_STATE, "operation: %s%s [%s%s]",
			desc.Description(), replaying ? " (replay)" : "",
			persistent ? "P" : "", synchronized ? "S" : "");
#endif

	delete access;

	}

