// $Id: Stats.h,v 1.4 2005/03/17 09:17:46 vern Exp $
//
// Classes that collect and report statistics.

#ifndef STATS_H
#define STATS_H

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "TCP_Endpoint.h"

class ProfileLogger {
public:
	ProfileLogger(BroFile* file, double interval);
	~ProfileLogger();

	void Log();
	BroFile* File()	{ return file; }

	void SegmentProfile(const char* name, const Location* loc,
				double dtime, int dmem);

private:
	BroFile* file;
	unsigned int log_count;
};

extern ProfileLogger* profiling_logger;
extern ProfileLogger* segment_logger;

// A SegmentProfiler tracks how much CPU and memory is consumed
// across its lifetime.
//
// ### This needs porting to Linux.  It could also be improved by
// better efforts at measuring its own overhead.
class SegmentProfiler {
public:
	// The constructor takes some way of identifying the segment.
	SegmentProfiler(ProfileLogger* arg_logger, const char* arg_name)
		{
		logger = arg_logger;
		if ( logger )
			{
			name = arg_name;
			loc = 0;
			Init();
			}
		}

	SegmentProfiler(ProfileLogger* arg_logger, const Location* arg_loc)
		{
		logger = arg_logger;
		if ( logger )
			{
			name = 0;
			loc = arg_loc;
			Init();
			}
		}

	~SegmentProfiler()
		{
		if ( logger )
			Report();
		}

protected:
	void Init();
	void Report();

	ProfileLogger* logger;
	const char* name;
	const Location* loc;
	struct rusage initial_rusage;
};

// A TCPStateStats object tracks the distribution of TCP states for
// the currently active connections.  
class TCPStateStats {
public:
	TCPStateStats();
	~TCPStateStats() { }

	void ChangeState(EndpointState o_prev, EndpointState o_now,
				EndpointState r_prev, EndpointState r_now);
	void FlipState(EndpointState orig, EndpointState resp);

	void StateEntered (EndpointState o_state, EndpointState r_state)
		{ ++state_cnt[o_state][r_state]; }
	void StateLeft (EndpointState o_state, EndpointState r_state)
		{ --state_cnt[o_state][r_state]; }

	unsigned int NumStateEstablished() const
		{ return state_cnt[TCP_ESTABLISHED][TCP_ESTABLISHED]; }
	unsigned int NumStateHalfClose() const
		{ // corresponds to S2,S3
		return state_cnt[TCP_ESTABLISHED][TCP_CLOSED] +
			state_cnt[TCP_CLOSED][TCP_ESTABLISHED];
		}
	unsigned int NumStateHalfRst() const
		{
		return state_cnt[TCP_ESTABLISHED][TCP_RESET] +
			state_cnt[TCP_RESET][TCP_ESTABLISHED];
		}
	unsigned int NumStateClosed() const
		{ return state_cnt[TCP_CLOSED][TCP_CLOSED]; }
	unsigned int NumStateRequest() const
		{
		assert(state_cnt[TCP_INACTIVE][TCP_SYN_SENT]==0);
		return state_cnt[TCP_SYN_SENT][TCP_INACTIVE];
		}
	unsigned int NumStateSuccRequest() const
		{
		return state_cnt[TCP_SYN_SENT][TCP_SYN_ACK_SENT] +
			state_cnt[TCP_SYN_ACK_SENT][TCP_SYN_SENT];
		}
	unsigned int NumStateRstRequest() const
		{
		return state_cnt[TCP_SYN_SENT][TCP_RESET] +
			state_cnt[TCP_RESET][TCP_SYN_SENT];
		}
	unsigned int NumStateInactive() const
		{ return state_cnt[TCP_INACTIVE][TCP_INACTIVE]; }
	unsigned int NumStatePartial() const;

	void PrintStats(BroFile* file, const char* prefix);

private:
	unsigned int state_cnt[TCP_RESET+1][TCP_RESET+1];
};

class PacketProfiler {
public:
	PacketProfiler(unsigned int mode, double freq, BroFile* arg_file);
	~PacketProfiler();

	static const unsigned int MODE_TIME = 1;
	static const unsigned int MODE_PACKET = 2;
	static const unsigned int MODE_VOLUME = 3;

	void ProfilePkt(double t, unsigned int bytes);

protected:
	BroFile* file;
	int update_mode;
	double update_freq;
	double last_Utime, last_Stime, last_Rtime;
	double last_timestamp, time;
	unsigned int last_mem;
	unsigned int pkt_cnt;
	unsigned int byte_cnt;
};

#endif
