// $Id: TCP.cc,v 1.13 2005/03/17 09:22:17 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.


#include "Active.h"
#include "File.h"
#include "TCP.h"
#include "TCP_Rewriter.h"
#include "OSFinger.h"
#include "Event.h"


// The following are not included in all systems' tcp.h.

#ifndef TH_ECE
#define TH_ECE  0x40
#endif

#ifndef TH_CWR
#define TH_CWR  0x80
#endif


#define TOO_LARGE_SEQ_DELTA 1048576


TCP_Connection::TCP_Connection(NetSessions* s, HashKey* k, double t,
		const ConnID* id, const struct tcphdr* tp)
: Connection(s, k, t, id)
	{
	// Set a timer to eventually time out this connection.
	ADD_TIMER(&TCP_Connection::ExpireTimer, t + tcp_SYN_timeout, 0,
			TIMER_TCP_EXPIRE);

	deferred_gen_event = close_deferred = 0;
	analyzer = 0;

	if ( ! (tp->th_flags & TH_SYN) || (tp->th_flags & TH_ACK) )
		is_partial = 1;
	else
		is_partial = 0;

	// By default, if it's a TCP port 80 FIN packet, don't record its
	// contents.  Eliminating these (unless we're doing HTTP analysis)
	// cuts the save file size by about a factor of two, since often
	// HTTP FIN packets have most of the server reply in them.
	if ( (tp->th_flags & TH_FIN) && ntohs(resp_port) == 80 )
		record_contents = 0;


	trace_rewriter = 0;
	src_pkt_writer = 0;
	}

void TCP_Connection::Init()
	{
	orig = new TCP_Endpoint(this, 1);
	resp = new TCP_Endpoint(this, 0);

	orig->SetPeer(resp);
	resp->SetPeer(orig);

	BuildEndpoints();

	// Can't put this in construction because RewritingTrace() is virtual.
	if ( transformed_pkt_dump && RewritingTrace() )
		{ // rewrite TCP trace
		trace_rewriter =
			new TCP_Rewriter(this, transformed_pkt_dump,
						transformed_pkt_dump_MTU,
						requires_trace_commitment);
		}

	if ( dump_selected_source_packets )
		{
		if ( source_pkt_dump )
			src_pkt_writer =
				new TCP_SourcePacketWriter(this, source_pkt_dump);
		else if ( transformed_pkt_dump )
			src_pkt_writer =
				new TCP_SourcePacketWriter(this, transformed_pkt_dump);
		}
	}

TCP_Connection::~TCP_Connection()
	{
	delete orig;
	delete resp;
	delete trace_rewriter;
	delete src_pkt_writer;
	}

void TCP_Connection::Done()
	{
	if ( trace_rewriter )
		trace_rewriter->Done();

	if ( tcp_match_undelivered )
		{
		orig->MatchUndeliveredData();
		resp->MatchUndeliveredData();
		}

	FinishEndpointMatcher();

	if ( connection_pending && is_active && ! BothClosed() )
		Event(connection_pending);

	while ( analyzer )
		{
		TCP_Analyzer* a = analyzer;
		analyzer = analyzer->NextAnalyzer();
		a->Done();
		Unref(a);
		}

	finished = 1;
	}

void TCP_Connection::AddAnalyzer(TCP_Analyzer* a)
	{
	a->AddAnalyzer(analyzer);
	analyzer = a;

	// Don't Ref(a) - when this connection goes away, we want
	// the analyzer to go away, too (unless somewhere else it was
	// Ref()'d).
	}

void TCP_Connection::AnalyzerDone()
	{
	TCP_Analyzer* a = analyzer;

	while ( a )
		{
		TCP_Analyzer* b = a;
		a = a->NextAnalyzer();
		b->Done();
		}

	// Don't Unref(b) - it'll be done when this connection goes away.
	}

void TCP_Connection::BuildEndpoints()
	{
	}

Val* TCP_Connection::BuildSYNPacketVal(int is_orig,
				const IP_Hdr* ip, const struct tcphdr* tcp)
	{
	int winscale = -1;
	int MSS = 0;
	int SACK = 0;

	// Parse TCP options.
	u_char* options = (u_char*) tcp + sizeof(struct tcphdr);
	u_char* opt_end = (u_char*) tcp + tcp->th_off * 4;

	while ( options < opt_end )
		{
		unsigned int opt = options[0];

		if ( opt == TCPOPT_EOL )
			// All done - could flag if more junk left over ....
			break;

		if ( opt == TCPOPT_NOP )
			{
			++options;
			continue;
			}

		if ( options + 1 >= opt_end )
			// We've run off the end, no room for the length.
			break;

		unsigned int opt_len = options[1];

		if ( options + opt_len > opt_end )
			// No room for rest of option.
			break;

		if ( opt_len == 0 )
			// Trashed length field.
			break;

		switch ( opt ) {
		case TCPOPT_SACK_PERMITTED:
			SACK = 1;
			break;

		case TCPOPT_MAXSEG:
			if ( opt_len < 4 )
				break;	// bad length

			MSS = (options[2] << 8) | options[3];
			break;

		case 3: // TCPOPT_WSCALE
			if ( opt_len < 3 )
				break;	// bad length

			winscale = options[2];
			break;

		default:	// just skip over
			break;
		}

		options += opt_len;
		}

	RecordVal* v = new RecordVal(SYN_packet);

	v->Assign(0, new Val(is_orig, TYPE_BOOL));
	v->Assign(1, new Val(int(ip->DF()), TYPE_BOOL));
	v->Assign(2, new Val(int(ip->TTL()), TYPE_INT));
	v->Assign(3, new Val((ip->TotalLen()), TYPE_INT));
	v->Assign(4, new Val(ntohs(tcp->th_win), TYPE_INT));
	v->Assign(5, new Val(winscale, TYPE_INT));
	v->Assign(6, new Val(MSS, TYPE_INT));
	v->Assign(7, new Val(SACK, TYPE_BOOL));

	return v;
	}

RecordVal* TCP_Connection::BuildOSVal(int is_orig, const IP_Hdr* ip,
			const struct tcphdr* tcp, uint32 tcp_hdr_len)
	{
	if ( ! is_orig )
		// Later we might use SYN-ACK fingerprinting here.
		return 0;

	// Passive OS fingerprinting wants to know a lot about IP and TCP
	// options: how many options there are, and in which order.
	int winscale = 0;
	int MSS = 0;
	int optcount = 0;
	uint32 quirks = 0;
	uint32 tstamp = 0;
	uint8 op[MAXOPT];

	if ( ip->HdrLen() > 20 )
		quirks |= QUIRK_IPOPT;

	if ( ip->IP_ID() == 0 )
		quirks |= QUIRK_ZEROID;

	if ( tcp->th_seq == 0 )
		quirks |= QUIRK_SEQ0;

	if ( tcp->th_seq == tcp->th_ack )
		quirks |= QUIRK_SEQEQ;

	if ( tcp->th_flags & ~(TH_SYN|TH_ACK|TH_RST|TH_ECE|TH_CWR) )
		quirks |= QUIRK_FLAGS;

	if ( ip->TotalLen() - ip->HdrLen() - tcp_hdr_len > 0 )
		quirks |= QUIRK_DATA;	// SYN with data

	if ( tcp->th_ack )
		quirks |= QUIRK_ACK;
	if ( tcp->th_urp )
		quirks |= QUIRK_URG;
	if ( tcp->th_x2 )
		quirks |= QUIRK_X2;

	// Parse TCP options.
	u_char* options = (u_char*) tcp + sizeof(struct tcphdr);
	u_char* opt_end = (u_char*) tcp + tcp_hdr_len;

	while ( options < opt_end )
		{
		unsigned int opt = options[0];

		if ( opt == TCPOPT_EOL )
			{
			op[optcount++] = TCPOPT_EOL;
			if ( ++options < opt_end )
				quirks |= QUIRK_PAST;

			// All done - could flag if more junk left over ....
			break;
			}

		if ( opt == TCPOPT_NOP )
			{
			op[optcount++] = TCPOPT_NOP;
			++options;
			continue;
			}

		if ( options + 1 >= opt_end )
			{
			// We've run off the end, no room for the length.
			quirks |= QUIRK_BROKEN;
			break;
			}

		unsigned int opt_len = options[1];

		if ( options + opt_len > opt_end )
			{
			// No room for rest of the options.
			quirks |= QUIRK_BROKEN;
			break;
			}

		if ( opt_len == 0 )
			// Trashed length field.
			break;

		switch ( opt ) {
		case TCPOPT_SACK_PERMITTED:
			// SACKOK LEN
			op[optcount] = TCPOPT_SACK_PERMITTED;
			break;

		case TCPOPT_MAXSEG:
			// MSS LEN D0 D1
			if ( opt_len < 4 )
				break;	// bad length

			op[optcount] = TCPOPT_MAXSEG;
			MSS = (options[2] << 8) | options[3];
			break;

		case TCPOPT_WINDOW:
			// WSCALE LEN D0
			if ( opt_len < 3 )
				break;	// bad length

			op[optcount] = TCPOPT_WINDOW;
			winscale = options[2];
			break;

		case TCPOPT_TIMESTAMP:
			// TSTAMP LEN T0 T1 T2 T3 A0 A1 A2 A3
			if ( opt_len < 10 )
				break;	// bad length

			op[optcount] = TCPOPT_TIMESTAMP;

			tstamp = ntohl(extract_uint32(options + 2));

			if ( extract_uint32(options + 6) )
				quirks |= QUIRK_T2;
			break;

		default:	// just skip over
			op[optcount]=opt;
			break;
		}

		if ( optcount < MAXOPT - 1 )
			++optcount;
		else
			quirks |= QUIRK_BROKEN;

		options += opt_len;
		}

	struct os_type os_from_print;
	int id = sessions->Get_OS_From_SYN(&os_from_print,
			uint16(ip->TotalLen()),
			uint8(ip->DF()), uint8(ip->TTL()),
			uint16(ntohs(tcp->th_win)),
			uint8(optcount), op,
			uint16(MSS), uint8(winscale),
			tstamp, quirks,
			uint8(tcp->th_flags & (TH_ECE|TH_CWR)));

	if ( sessions->CompareWithPreviousOSMatch(ip->SrcAddr4(), id) )
		{
		RecordVal* os = new RecordVal(OS_version);

		os->Assign(0, new StringVal(os_from_print.os));

		if ( os_from_print.desc )
			os->Assign(1, new StringVal(os_from_print.desc));
		else
			os->Assign(1, new StringVal(""));

		os->Assign(2, new Val(os_from_print.dist, TYPE_COUNT));
		os->Assign(3, new EnumVal(os_from_print.match, OS_version_inference));

		return os;
		}

	return 0;
	}

int TCP_Connection::ParseTCPOptions(const struct tcphdr* tcp,
				    proc_tcp_option_t proc,
				    TCP_Connection* conn,
				    int is_orig,
				    void* cookie)
	{
	// Parse TCP options.
	const u_char* options = (const u_char*) tcp + sizeof(struct tcphdr);
	const u_char* opt_end = (const u_char*) tcp + tcp->th_off * 4;

	while ( options < opt_end )
		{
		unsigned int opt = options[0];

		unsigned int opt_len;

		if ( opt < 2 )
			opt_len = 1;

		else if ( options + 1 >= opt_end )
			// We've run off the end, no room for the length.
			return -1;

		else
			opt_len = options[1];

		if ( opt_len == 0 )
			return -1;	// trashed length field

		if ( options + opt_len > opt_end )
			// No room for rest of option.
			return -1;

		if ( (*proc)(opt, opt_len, options, conn, is_orig, cookie) == -1 )
			return -1;

		options += opt_len;

		if ( opt == TCPOPT_EOL )
			// All done - could flag if more junk left over ....
			break;
		}

	return 0;
	}

int TCP_Connection::TCPOptionEvent(unsigned int opt,
				   unsigned int optlen,
				   const u_char* /* option */,
				   TCP_Connection* conn,
				   int is_orig,
				   void* cookie)
	{
	if ( tcp_option )
		{
		val_list* vl = new val_list();

		vl->append(conn->BuildConnVal());
		vl->append(new Val(is_orig, TYPE_BOOL));
		vl->append(new Val(opt, TYPE_COUNT));
		vl->append(new Val(optlen, TYPE_COUNT));

		conn->ConnectionEvent(tcp_option, vl);
		}

	return 0;
	}

void TCP_Connection::NextPacket(double t, int is_orig,
			const IP_Hdr* ip,
			int len, int caplen, const u_char*& data,
			int& record_packet, int& record_content,
			const struct pcap_pkthdr* pcap_hdr,
			const u_char* const pcap_pkt,
			int hdr_size)
	{
	const struct tcphdr* tp = (const struct tcphdr*) data;
	uint32 tcp_hdr_len = tp->th_off * 4;

	if ( tcp_hdr_len < sizeof(struct tcphdr) )
		{
		Weird("bad_TCP_header_len");
		return;
		}

	if ( tcp_hdr_len > uint32(len) )
		{
		// This can happen even with the above test, due to TCP
		// options.
		Weird("truncated_header");
		return;
		}

	len -= tcp_hdr_len;	// remove TCP header
	caplen -= tcp_hdr_len;
	data += tcp_hdr_len;

	TCP_Endpoint* endpoint = is_orig ? orig : resp;
	TCP_Endpoint* peer = endpoint->peer;

	if ( ! ignore_checksums && caplen >= len &&
	     ! endpoint->ValidChecksum(tp, len) )
		{
		Weird("bad_TCP_checksum");
		return;
		}

	if ( rule_matcher )
		{
		if ( is_orig && ! orig_match_state )
			InitEndpointMatcher(ip, caplen, 1);

		if ( ! is_orig && ! resp_match_state )
			InitEndpointMatcher(ip, caplen, 0);
		}

	u_char flags = tp->th_flags;
	uint32 base_seq = ntohl(tp->th_seq);
	uint32 ack_seq = ntohl(tp->th_ack);
	int SYN = flags & TH_SYN;
	int FIN = flags & TH_FIN;
	int RST = flags & TH_RST;
	int ACK = flags & TH_ACK;
	int seq_len = len;	// length in terms of sequence space

	// Note, the offered window on an initial SYN is unscaled, even
	// if the SYN includes scaling, so we need to do the following
	// test *before* updating the scaling information below.  (Hmmm,
	// how does this work for windows on SYN/ACKs? ###)
	int scale = endpoint->window_scale;
	unsigned int window = (ntohs(tp->th_win)) << scale;

	if ( ! SYN )
		{
		// Don't analyze window values off of SYNs, they're sometimes
		// immediately rescinded.

		// ### Decide whether to accept new window based on AM policy.
		if ( int(base_seq - endpoint->window_seq) >= 0 &&
		     int(ack_seq - endpoint->window_ack_seq) >= 0 )
			{
			uint32 new_edge = ack_seq + window;
			uint32 old_edge = endpoint->window_ack_seq + endpoint->window;
			int advance = new_edge - old_edge;

			if ( advance < 0 )
				{
				// A window recision.  We don't report these
				// for FINs or RSTs, or if the connection
				// has already been partially closed, since
				// such recisions occur frequently in practice,
				// probably as the receiver loses buffer memory
				// due to its process going away.
				//
				// We also, for window scaling, allow a bit
				// of slop ###.  This is because sometimes
				// there will be an apparent recision due
				// to the granularity of the scaling.
				if ( ! FIN && ! RST &&
				     endpoint->state != TCP_CLOSED &&
				     endpoint->state != TCP_RESET &&
				     (-advance) >= (1 << scale) )
					Weird("window_recision");
				}

			endpoint->window = window;
			endpoint->window_ack_seq = ack_seq;
			endpoint->window_seq = base_seq;
			}
		}

	if ( ! orig->did_close || ! resp->did_close )
		last_time = t;

	if ( SYN )
		{
		++seq_len;	// SYN consumes a byte of sequence space
		if ( RST )
			Weird("TCP_christmas");

		if ( (flags & TH_URG) )
			Weird("baroque_SYN");

		if ( len > 0 )
			// T/TCP definitely complicates this.
			Weird("SYN_with_data");

		RecordVal* SYN_vals =
			BuildSYNPacketVal(is_orig, ip, tp)->AsRecordVal();

		// ### In the following, we could be fooled by an
		// inconsistent SYN retransmission.  Where's a normalizer
		// when you need one?

		// ### We know that field 5 is the window scaling ....
		int scale = SYN_vals->Lookup(5)->CoerceToInt();

		if ( scale < 0 )
			{ // no window scaling option
			if ( ACK )
				{ // window scaling not negotiated
				endpoint->window_scale = 0;
				peer->window_scale = 0;
				}
			else
				// We're not offering window scaling.
				// Ideally, we'd remember this fact so that
				// if the SYN/ACK *does* include window
				// scaling, we know it won't be negotiated.
				// But it's a pain to track that, and hard
				// to see how an adversarial responder could
				// use it to evade.  Also, if we *do* want
				// to track it, we could do so using
				// connection_SYN_packet.
				endpoint->window_scale = 0;
			}
		else
			{
			endpoint->window_scale = scale;
			endpoint->window_seq = base_seq;
			endpoint->window_ack_seq = ack_seq;

			peer->window_seq = ack_seq;
			peer->window_ack_seq = base_seq;
			}

		if ( connection_SYN_packet )
			{
			val_list* vl = new val_list;
			vl->append(BuildConnVal());
			vl->append(SYN_vals);
			ConnectionEvent(connection_SYN_packet, vl);
			}
		else
			Unref(SYN_vals);

		// Passive fingerprinting.
		//
		// is_orig will be removed once we can do SYN-ACK
		// fingerprinting.
		if ( OS_version_found && is_orig )
			{
			Val src_addr_val(*OrigAddr(), TYPE_ADDR);
			if ( generate_OS_version_event->Size() == 0 ||
			     generate_OS_version_event->Lookup(&src_addr_val) )
				{
				RecordVal* OS_val =
					BuildOSVal(is_orig, ip, tp, tcp_hdr_len);
				if ( OS_val )
					{ // found new OS version
					val_list* vl = new val_list;
					vl->append(BuildConnVal());
					vl->append(new AddrVal(OrigAddr()));
					vl->append(OS_val);
					ConnectionEvent(OS_version_found, vl);
					}
				}
			}
		}

	if ( FIN )
		{
		++seq_len;	// FIN consumes a byte of sequence space
		// Remember the relative seq in FIN_seq.
		endpoint->FIN_seq = base_seq - endpoint->start_seq + seq_len;

		if ( t < endpoint->last_time + tcp_storm_interarrival_thresh &&
		     ++endpoint->FIN_cnt == tcp_storm_thresh )
			Weird("FIN_storm");

		else if ( endpoint->FIN_cnt == 0 )
			// Remember that we've seen a FIN.
			++endpoint->FIN_cnt;
		}

	if ( RST )
		{
		if ( t < endpoint->last_time + tcp_storm_interarrival_thresh &&
		     ++endpoint->RST_cnt == tcp_storm_thresh )
			Weird("RST_storm");

		else if ( endpoint->RST_cnt == 0 )
			++endpoint->RST_cnt;	// Remember we've seen a RST

#ifdef ACTIVE_MAPPING
//		if ( peer->state == TCP_INACTIVE )
//		    debug_msg("rst while inactive\n"); // ### Cold-start: what to do?
//		else

		const NumericData* AM_policy;
		get_map_result(ip->DstAddr4(), AM_policy);

		if ( base_seq == endpoint->ack_seq )
			;	 // everyone should accept a RST in sequence

		else if ( endpoint->window == 0 )
			; // ### Cold Start: we don't know the window,
			  // so just go along for now

		else
			{
			uint32 right_edge = endpoint->ack_seq +
				(endpoint->window << endpoint->window_scale);

			if ( base_seq < right_edge  )
				{
				if ( ! AM_policy->accepts_rst_in_window )
					{
#if 0
					debug_msg("Machine does not accept RST merely in window; ignoring. t=%.6f,base=%u, ackseq=%u, window=%hd \n",
						network_time, base_seq, endpoint->ack_seq, window);
#endif
					return;
					}
				}

			else if ( ! AM_policy->accepts_rst_outside_window )
				{
#if 0
				debug_msg("Machine does not accept RST outside window; ignoring. t=%.6f,base=%u, ackseq=%u, window=%hd \n",
					network_time, base_seq, endpoint->ack_seq, window);
#endif
				return;
				}
			}
#endif

		if ( len > 0 )
			{
			// This now happens often enough that it's
			// not in the least interesting.
			// Weird("RST_with_data");

			// Don't include the data in the computation of
			// the sequence space for this connection, as
			// it's not in fact part of the TCP stream.
			seq_len = 0;
			}

		PacketWithRST();
		}

	uint32 last_seq = base_seq + seq_len;

	if ( endpoint->state == TCP_INACTIVE )
		{
		if ( ! SYN )
			// This is a partial connection - set up the
			// initial sequence numbers as though we saw
			// a SYN, to keep the relative byte numbering
			// consistent.
			endpoint->ack_seq = endpoint->start_seq = base_seq - 1;
		else
			endpoint->ack_seq = endpoint->start_seq = base_seq;

		endpoint->last_seq = last_seq;
		endpoint->start_time = t;
		}

	int delta_last = seq_delta(last_seq, endpoint->last_seq);

	if ( (SYN || RST) && (delta_last > TOO_LARGE_SEQ_DELTA || delta_last < -TOO_LARGE_SEQ_DELTA) )
		// ### perhaps trust RST seq #'s if initial and not too
		// outlandish, but not if they're coming after the other
		// side has sent a FIN - trust the FIN ack instead
		;

	else if ( FIN && endpoint->last_seq == endpoint->start_seq + 1 )
		// Update last_seq based on the FIN even if delta_last < 0.
		// This is to accommodate > 2 GB connections for which
		// we've only seen the SYN and the FIN (hence the check
		// for last_seq == start_seq + 1).
		endpoint->last_seq = last_seq;

	else if ( endpoint->state == TCP_RESET )
		// don't trust any subsequent sequence numbers
		;

	else if ( delta_last > 0 )
		// ### check for large jumps here.
		endpoint->last_seq = last_seq;

	else if ( delta_last <= 0 )
		// ### ++retransmit, unless this is a pure ack
		;

	if ( ACK )
		{
		if ( peer->state == TCP_INACTIVE )
			{
			if ( ! SYN && ! FIN && ! RST )
				{
				if ( endpoint->state == TCP_SYN_SENT ||
				     endpoint->state == TCP_SYN_ACK_SENT ||
				     endpoint->state == TCP_ESTABLISHED )
					{
					// We've already sent a SYN, but
					// that hasn't roused the other
					// end, yet we're ack'ing their
					// data.
					if ( weird == 0 )
						Weird("possible_split_routing");
					}
				}

			// Start the sequence numbering as if there
			// was an initial SYN, so the relative
			// numbering of subsequent data packets stays
			// consistent.
			peer->start_seq = peer->ack_seq = peer->last_seq =
				ack_seq - 1;
			}

		else if ( ! RST )
			{ // don't trust ack's in RST packets
			int delta_ack = seq_delta(ack_seq, peer->ack_seq);
			if ( ack_seq == 0 && delta_ack > TOO_LARGE_SEQ_DELTA )
				// More likely that this is a broken ack
				// than a large connection that happens
				// to land on 0 in the sequence space.
				;

			else if ( delta_ack > 0 )
				peer->ack_seq = ack_seq;
			}

		peer->AckReceived(ack_seq - peer->start_seq);
		}

	endpoint->last_time = t;

	int do_close = 0;	// whether to report the connection as closed
	int gen_event = 0;	// if so, whether to generate an event

	switch ( endpoint->state ) {

	case TCP_INACTIVE:
		{
		if ( SYN )
			{
			if ( is_orig )
				{
				if ( ACK )
					{
					Weird("connection_originator_SYN_ack");
					endpoint->SetState(TCP_SYN_ACK_SENT);
					}
				else
					endpoint->SetState(TCP_SYN_SENT);

				if ( connection_attempt )
					ADD_TIMER(&TCP_Connection::AttemptTimer,
						t + tcp_attempt_delay, 1,
						TIMER_TCP_ATTEMPT);
				}
			else
				{
				if ( ACK )
					{
					if ( peer->state != TCP_INACTIVE &&
					     peer->state != TCP_PARTIAL &&
					     ! seq_between(ack_seq, peer->start_seq, peer->last_seq) )
						Weird("bad_SYN_ack");
					}

				else if ( peer->state == TCP_SYN_ACK_SENT &&
					  base_seq == endpoint->start_seq )
					{
					// This is a SYN/SYN-ACK reversal,
					// per the discussion in IsReuse.
					// Flip the endpoints and establish
					// the connection.

					// Flip state statistics so that the
					// later SetState works as it should.
					sessions->tcp_stats.FlipState(orig->state, resp->state);
					TCP_Endpoint* tmp_ep = resp;
					resp = orig;
					orig = tmp_ep;

					uint32 tmp_addr[NUM_ADDR_WORDS];
					copy_addr(resp_addr, tmp_addr);
					copy_addr(orig_addr, resp_addr);
					copy_addr(tmp_addr, orig_addr);

					uint16 tmp_port = resp_port;
					resp_port = orig_port;
					orig_port = tmp_port;

					Unref(conn_val);
					conn_val = 0;

					RuleEndpointState* tmp = orig_match_state;
					orig_match_state = resp_match_state;
					resp_match_state = tmp;
					if ( orig_match_state )
						orig_match_state->FlipIsOrig();
					if ( resp_match_state )
						resp_match_state->FlipIsOrig();

					peer->SetState(TCP_ESTABLISHED);
					}

				else
					Weird("simultaneous_open");

				if ( peer->state == TCP_SYN_SENT )
					peer->SetState(TCP_ESTABLISHED);
				else if ( peer->state == TCP_INACTIVE )
					{
					// If we were to ignore SYNs and
					// only instantiate state on SYN
					// acks, then we'd do:
					//    peer->SetState(TCP_ESTABLISHED);
					// here.
					Weird("unsolicited_SYN_response");
					}

				endpoint->SetState(TCP_ESTABLISHED);

				if ( peer->state != TCP_PARTIAL )
					{
					Event(connection_established);
					EnableStatusUpdateTimer();
					}
				}
			}

		if ( FIN )
			{
			endpoint->SetState(TCP_CLOSED);
			do_close = gen_event = 1;
			if ( peer->state != TCP_PARTIAL && ! SYN )
				Weird("spontaneous_FIN");
			}

		if ( RST )
			{
			endpoint->SetState(TCP_RESET);

			int is_reject = 0;

			if ( is_orig )
				{
				// If our peer is established then we saw
				// a SYN-ack but not SYN - so a reverse
				// scan, and we should treat this as a
				// reject.
				if ( peer->state == TCP_ESTABLISHED )
					is_reject = 1;
				}

			else if ( peer->state == TCP_SYN_SENT ||
				  peer->state == TCP_SYN_ACK_SENT )
				// We're rejecting an initial SYN.
				is_reject = 1;

			do_close = 1;
			gen_event = ! is_reject;

			if ( is_reject )
				Event(connection_rejected);

			else if ( peer->state == TCP_INACTIVE )
				Weird("spontaneous_RST");
			}

		if ( endpoint->state == TCP_INACTIVE )
			{ // No control flags to change the state.
			if ( ! is_orig && len == 0 &&
			     orig->state == TCP_SYN_SENT )
				// Some eccentric TCP's will ack an initial
				// SYN prior to sending a SYN reply (hello,
				// ftp.microsoft.com).  For those, don't
				// consider the ack as forming a partial
				// connection.
				;
			else
				{
				endpoint->SetState(TCP_PARTIAL);
				EnableStatusUpdateTimer();
				
				if ( peer->state == TCP_PARTIAL )
					// We've seen both sides of a partial
					// connection, report it.
					Event(partial_connection);
				}
			}
		}
		break;

	case TCP_SYN_SENT:
	case TCP_SYN_ACK_SENT:
		{
		if ( SYN )
			{
			if ( is_orig )
				{
				if ( ACK && ! FIN && ! RST &&
				     endpoint->state != TCP_SYN_ACK_SENT )
					Weird("repeated_SYN_with_ack");
				else if ( peer->state == TCP_INACTIVE )
					// ###
					start_time = endpoint->start_time = t;
				}
			else
				{
				if ( ! ACK &&
				     endpoint->state != TCP_SYN_SENT )
					Weird("repeated_SYN_reply_wo_ack");
				}

			if ( base_seq != endpoint->start_seq )
				{
				Weird("SYN_seq_jump");
				endpoint->ack_seq = endpoint->start_seq = base_seq;
				endpoint->last_seq = last_seq;
				}
			}

		if ( FIN )
			{
			if ( peer->state == TCP_INACTIVE ||
			     peer->state == TCP_SYN_SENT )
				Weird("inappropriate_FIN");
			endpoint->SetState(TCP_CLOSED);
			do_close = gen_event = 1;
			}

		if ( RST )
			{
			endpoint->SetState(TCP_RESET);
			ConnectionReset();
			do_close = 1;
			}
		else if ( len > 0 )
			Weird("data_before_established");
		}
		break;

	case TCP_ESTABLISHED:
	case TCP_PARTIAL:
		{
		if ( SYN )
			{
			if ( endpoint->state == TCP_PARTIAL &&
			     peer->state == TCP_INACTIVE && ! ACK )
				{
				Weird("SYN_after_partial");
				endpoint->SetState(TCP_SYN_SENT);
				}

			if ( endpoint->Size() > 0 )
				Weird("SYN_inside_connection");

			if ( is_orig && peer->state == TCP_ESTABLISHED ) // ###
				start_time = endpoint->start_time = t;

			if ( base_seq != endpoint->start_seq )
				Weird("SYN_seq_jump");

			// Make a guess that somehow the connection didn't
			// get established, and this SYN will be the
			// one that actually sets it up.
			endpoint->ack_seq = endpoint->start_seq = base_seq;
			endpoint->last_seq = last_seq;
			}

		if ( FIN && ! RST )	// ###
			{ // should check sequence/ack numbers here ###
			endpoint->SetState(TCP_CLOSED);

			if ( peer->state == TCP_RESET &&
			     peer->prev_state == TCP_CLOSED )
				// The peer sent a FIN followed by a RST.
				// Turn it back into CLOSED state, because
				// this was actually normal termination.
				peer->SetState(TCP_CLOSED);

			do_close = gen_event = 1;
			}

		if ( RST )
			{
			endpoint->SetState(TCP_RESET);
			do_close = 1;

			if ( peer->state != TCP_RESET ||
			     peer->prev_state != TCP_ESTABLISHED )
				ConnectionReset();
			}
		}
		break;

	case TCP_CLOSED:
		{
		if ( SYN )
			Weird("SYN_after_close");

		if ( FIN && delta_last > 0 )
			// Probably should also complain on FIN recision.
			// That requires an extra state variable to avoid
			// generating slews of weird's when a TCP gets
			// seriously confused (this from experience).
			Weird("FIN_advanced_last_seq");

		if ( RST )
			{
			endpoint->SetState(TCP_RESET);

			if ( ! endpoint->did_close )
				// RST after FIN.
				do_close = 1;

			if ( connection_reset )
				ADD_TIMER(&TCP_Connection::ResetTimer,
						t + tcp_reset_delay, 1,
						TIMER_TCP_RESET);
			}
		}
		break;

	case TCP_RESET:
		{
		if ( SYN )
			Weird("SYN_after_reset");
		if ( FIN )
			Weird("FIN_after_reset");

		if ( len > 0 && ! RST )
			Weird("data_after_reset");
		}
		break;
	}

	if ( tcp_packet )
		{
		char tcp_flags[256];
		int tcp_flag_len = 0;

		if ( SYN ) tcp_flags[tcp_flag_len++] = 'S';
		if ( FIN ) tcp_flags[tcp_flag_len++] = 'F';
		if ( RST ) tcp_flags[tcp_flag_len++] = 'R';
		if ( ACK ) tcp_flags[tcp_flag_len++] = 'A';
		if ( flags & TH_PUSH ) tcp_flags[tcp_flag_len++] = 'P';
		if ( flags & TH_URG ) tcp_flags[tcp_flag_len++] = 'U';

		tcp_flags[tcp_flag_len] = '\0';

		val_list* vl = new val_list();

		vl->append(BuildConnVal());
		vl->append(new Val(is_orig, TYPE_BOOL));
		vl->append(new StringVal(tcp_flags));
		vl->append(new Val(base_seq - endpoint->start_seq, TYPE_COUNT));
		vl->append(new Val(ACK ? ack_seq - peer->start_seq : 0, TYPE_COUNT));
		vl->append(new Val(len, TYPE_COUNT));
		vl->append(new StringVal(caplen, (const char*) data));

		mgr.QueueEvent(tcp_packet, vl);
		}

	if ( tcp_option && tcp_hdr_len > sizeof(*tp) )
		ParseTCPOptions(tp, TCPOptionEvent, this, is_orig, 0);

	if ( trace_rewriter && pcap_hdr )
		trace_rewriter->NextPacket(is_orig, t, pcap_hdr,
					pcap_pkt, hdr_size, ip->IP4_Hdr(), tp);

	if ( src_pkt_writer && pcap_hdr )
		src_pkt_writer->NextPacket(pcap_hdr, pcap_pkt);

	int need_contents = 0;
	if ( len > 0 && (caplen >= len || endpoint->Analyzer()) &&
	     ! RST && ! skip )
		{
		int data_seq = base_seq - endpoint->start_seq;
		if ( SYN )
			++data_seq;	// skip over SYN octet

		need_contents = endpoint->DataSent(t, data_seq,
						min(len, caplen), data, ip, tp);
		}

	endpoint->CheckEOF();

	if ( do_close )
		{
		// We need to postpone doing this until after we process
		// DataSent, so we don't generate a connection_finished event
		// until after data perhaps included with the FIN is processed.
		ConnectionClosed(endpoint, peer, gen_event);
		}

	record_content = need_contents || RecordContents();
//###	record_content = RecordContents();	// a hack to avoid dumping EVERY packet
	record_packet = record_packets || SYN || FIN || RST;
	}

void TCP_Connection::Deliver(TCP_Endpoint* /* sender */,
			int /* seq */, int /* len */, u_char* /* data */)
	{
	}

void TCP_Connection::Undelivered(TCP_Endpoint* sender, int seq, int len)
	{
	// This can happen because we're processing a trace that's been
	// filtered.  For example, if it's just SYN/FIN data, then there
	// can be data in the FIN packet, but it's undelievered because
	// it's out of sequence.
	if ( content_gap )
		{
		val_list* vl = new val_list;
		vl->append(BuildConnVal());
		vl->append(new Val(IsOrig(sender), TYPE_BOOL));
		vl->append(new Val(seq, TYPE_COUNT));
		vl->append(new Val(len, TYPE_COUNT));

		ConnectionEvent(content_gap, vl);
		}

	if ( trace_rewriter )
		trace_rewriter->ContentGap(IsOrig(sender), len);
	}

void TCP_Connection::NewLine(TCP_ContentLine* /* sender */, int /* length */, const char* /* line */)
	{
	}

void TCP_Connection::SetEnv(int /* is_orig */, char* name, char* val)
	{
	delete [] name;
	delete [] val;
	}

void TCP_Connection::AttemptTimer(double /* t */)
	{
	if ( ! is_active )
		return;

	if ( (orig->state == TCP_SYN_SENT || orig->state == TCP_SYN_ACK_SENT) &&
	     resp->state == TCP_INACTIVE )
		{
		Event(connection_attempt);
		is_active = 0;

		// All done with this connection.
		sessions->Remove(this);
		}
	}

void TCP_Connection::PartialCloseTimer(double /* t */)
	{
	if ( ! is_active )
		return;

	if ( orig->state != TCP_INACTIVE && resp->state != TCP_INACTIVE &&
	     (! orig->did_close || ! resp->did_close) )
		{
		if ( orig->state == TCP_RESET || resp->state == TCP_RESET )
			// Presumably the RST is what caused the partial
			// close.  Don't report it.
			return;

		Event(connection_partial_close);
		sessions->Remove(this);
		}
	}

void TCP_Connection::ExpireTimer(double t)
	{
	if ( ! is_active )
		return;

	if ( last_time + tcp_connection_linger < t )
		{
		if ( orig->did_close || resp->did_close )
			{
			// No activity for tcp_connection_linger seconds, and
			// at least one side has closed.  See whether
			// connection has likely terminated.
			if ( (orig->did_close && resp->did_close) ||
			     (orig->state == TCP_RESET ||
			      resp->state == TCP_RESET) ||
			     (orig->state == TCP_INACTIVE ||
			      resp->state == TCP_INACTIVE) )
				{
				// Either both closed, or one RST,
				// or half-closed.

				// The Timer has Ref()'d us and won't Unref()
				// us until we return, so it's safe to have
				// the session remove and Unref() us here.
				Event(connection_timeout);
				is_active = 0;
				sessions->Remove(this);
				return;
				}
			}

		if ( resp->state == TCP_INACTIVE )
			{
			if ( (orig->state == TCP_SYN_SENT ||
			      orig->state == TCP_SYN_ACK_SENT) )
				{
				if ( ! connection_attempt )
					{
					// Time out the connection attempt,
					// since the AttemptTimer isn't going
					// to do it for us, and we don't want
					// to clog the data structures with
					// old, failed attempts.
					Event(connection_timeout);
					is_active = 0;
					sessions->Remove(this);
					return;
					}
				}

			else if ( orig->state == TCP_INACTIVE )
				{
				// Nothing ever happened on this connection.
				// This can occur when we see a trashed
				// packet - it's discarded by NextPacket
				// before setting up an attempt timer,
				// so we need to clean it up here.
				Event(connection_timeout);
				sessions->Remove(this);
				return;
				}
			}
		}

	// Connection still active, so reschedule timer.
	// ### if PQ_Element's were BroObj's, could just Ref the timer
	// and adjust its value here, instead of creating a new timer.
	ADD_TIMER(&TCP_Connection::ExpireTimer, t + tcp_session_timer,
			0, TIMER_TCP_EXPIRE);
	}

void TCP_Connection::ResetTimer(double /* t */)
	{
	if ( ! is_active )
		return;

	if ( ! BothClosed() )
		ConnectionReset();

	sessions->Remove(this);
	}

void TCP_Connection::DeleteTimer(double /* t */)
	{
	sessions->Remove(this);
	}


// The following need to be consistent with bro.init.
#define CONTENTS_NONE 0
#define CONTENTS_ORIG 1
#define CONTENTS_RESP 2
#define CONTENTS_BOTH 3

void TCP_Connection::SetContentsFile(unsigned int direction, BroFile* f)
	{
	if ( direction == CONTENTS_NONE )
		{
		orig->SetContentsFile(0);
		resp->SetContentsFile(0);
		}

	else
		{
		if ( direction == CONTENTS_ORIG || direction == CONTENTS_BOTH )
			orig->SetContentsFile(f);
		if ( direction == CONTENTS_RESP || direction == CONTENTS_BOTH )
			resp->SetContentsFile(f);

		if ( direction == CONTENTS_BOTH )
			Ref(f);	// since it was handed off twice above
		}
	}

BroFile* TCP_Connection::GetContentsFile(unsigned int direction) const
	{
	switch ( direction ) {
	case CONTENTS_NONE:
		return 0;

	case CONTENTS_ORIG:
		return orig->GetContentsFile();

	case CONTENTS_RESP:
		return resp->GetContentsFile();

	case CONTENTS_BOTH:
		if ( orig->GetContentsFile() != resp->GetContentsFile())
			// This is an "error".
			return 0;
		else
			return orig->GetContentsFile();

	default:
		Internal("inconsistency in TCP_Connection::GetContentsFile");
	}
	}

void TCP_Connection::SetTraceRewriter(TCP_Rewriter* rewriter)
	{
	trace_rewriter = rewriter;
	}

int TCP_Connection::IsReuse(double t, const u_char* pkt)
	{
	const struct tcphdr* tp = (const struct tcphdr*) pkt;

	if ( unsigned(tp->th_off) < sizeof(struct tcphdr) / 4 )
		// Bogus header, don't interpret further.
		return 0;

	TCP_Endpoint* conn_orig = orig;

	// Reuse only occurs on initial SYN's, except for half connections
	// it can occur on SYN-acks.
	if ( ! (tp->th_flags & TH_SYN) )
		return 0;

	if ( (tp->th_flags & TH_ACK) )
		{
		if ( orig->state != TCP_INACTIVE )
			// Not a half connection.
			return 0;

		conn_orig = resp;
		}

	if ( ! IsClosed() )
		{
		uint32 base_seq = ntohl(tp->th_seq);
		if ( base_seq == conn_orig->start_seq )
			return 0;

		if ( (tp->th_flags & TH_ACK) == 0 &&
		     conn_orig->state == TCP_SYN_ACK_SENT &&
		     resp->state == TCP_INACTIVE &&
		     base_seq == resp->start_seq )
			{
			// This is an initial SYN with the right sequence
			// number, and the state is consistent with the
			// SYN & the SYN-ACK being flipped (e.g., due to
			// reading from two interfaces w/ interrupt
			// coalescence).  Don't treat this as a reuse.
			// NextPacket() will flip set the connection
			// state correctly
			return 0;
			}

		if ( conn_orig->state == TCP_SYN_SENT )
			Weird("SYN_seq_jump");
		else
			Weird("active_connection_reuse");
		}

	else if ( (orig->IsActive() || resp->IsActive()) &&
	     orig->state != TCP_RESET && resp->state != TCP_RESET )
		Weird("active_connection_reuse");

	else if ( t - last_time < tcp_connection_linger &&
	     orig->state != TCP_RESET && resp->state != TCP_RESET )
		Weird("premature_connection_reuse");

	return 1;
	}

void TCP_Connection::EndpointEOF(TCP_Contents* endp)
	{
	if ( connection_EOF )
		{
		val_list* vl = new val_list();
		vl->append(BuildConnVal());
		vl->append(new Val(endp->IsOrig(), TYPE_BOOL));
		ConnectionEvent(connection_EOF, vl);
		}

	TraceRewriterEOF(endp->Endpoint());

	if ( close_deferred )
		{
		if ( DataPending(endp->Endpoint()) )
			{
			if ( BothClosed() )
				Weird("pending_data_when_closed");

			// Defer further, until the other endpoint
			// EOF's, too.
			}

		ConnectionClosed(endp->Endpoint(), endp->Endpoint()->peer,
					deferred_gen_event);
		close_deferred = 0;
		}
	}

void TCP_Connection::TraceRewriterEOF(TCP_Endpoint* endp)
	{
	// Add a FIN packet if there is one in the original trace.
	if ( trace_rewriter && endp->FIN_cnt > 0 )
		trace_rewriter->ScheduleFIN(IsOrig(endp));
	}

void TCP_Connection::Describe(ODesc* d) const
	{
	d->Add(start_time);
	d->Add("(");
	d->Add(last_time);
	d->AddSP(")");

	d->Add(dotted_addr(orig_addr));
	d->Add(".");
	d->Add(ntohs(orig_port));
	if ( ! orig->did_close )
		d->Add("*");

	d->SP();
	d->AddSP("->");

	d->Add(dotted_addr(resp_addr));
	d->Add(".");
	d->Add(ntohs(resp_port));
	if ( ! resp->did_close )
		d->Add("*");

	d->SP();

	d->Add("(");
	d->Add(orig->Size());
	d->AddSP(",");
	d->Add(resp->Size());
	d->Add(")");
	d->NL();
	}

void TCP_Connection::ConnectionClosed(TCP_Endpoint* endpoint, TCP_Endpoint* peer,
					int gen_event)
	{
	if ( DataPending(endpoint) )
		{
		// Don't close out the connection yet, there's still data to
		// deliver.
		close_deferred = 1;
		if ( ! deferred_gen_event )
			deferred_gen_event = gen_event;
		return;
		}

	close_deferred = 0;

	if ( endpoint->did_close )
		return;	// nothing new to report

	endpoint->did_close = 1;

	int close_complete = endpoint->state == TCP_RESET ||
			     peer->did_close || peer->state == TCP_INACTIVE;

	if ( close_complete )
		{
		if ( endpoint->prev_state != TCP_INACTIVE ||
		     peer->state != TCP_INACTIVE )
			{
			if ( deferred_gen_event )
				{
				gen_event = 1;
				deferred_gen_event = 0;	// clear flag
				}

			// We have something interesting to report.
			if ( gen_event )
				{
				if ( peer->state == TCP_INACTIVE )
					ConnectionFinished(1);
				else
					ConnectionFinished(0);
				}
			}

		CancelTimers();

		// Note, even if tcp_close_delay is zero, we can't
		// simply do:
		//
		//	sessions->Remove(this);
		//
		// here, because that would cause the object to be
		// deleted out from under us.
		if ( tcp_close_delay != 0.0 )
			ADD_TIMER(&TCP_Connection::ConnDeleteTimer,
				last_time + tcp_close_delay, 0,
				TIMER_CONN_DELETE);
		else
			ADD_TIMER(&TCP_Connection::DeleteTimer, last_time, 0,
					TIMER_TCP_DELETE);
		}

	else
		{ // We haven't yet seen a full close.
		if ( endpoint->prev_state == TCP_INACTIVE )
			{ // First time we've seen anything from this side.
			if ( connection_partial_close )
				ADD_TIMER(&TCP_Connection::PartialCloseTimer,
					last_time + tcp_partial_close_delay, 0,
					TIMER_TCP_PARTIAL_CLOSE );
			}

		else
			{
			// Create a timer to look for the other side closing,
			// too.
			ADD_TIMER(&TCP_Connection::ExpireTimer,
					last_time + tcp_session_timer, 0,
					TIMER_TCP_EXPIRE);
			}
		}
	}

void TCP_Connection::ConnectionFinished(int half_finished)
	{
	if ( half_finished )
		Event(connection_half_finished);
	else
		Event(connection_finished);

	is_active = 0;
	}

void TCP_Connection::ConnectionReset()
	{
	Event(connection_reset);
	is_active = 0;
	}

int TCP_Connection::DataPending(TCP_Endpoint* closing_endp)
	{
	if ( Skipping() )
		return 0;

	return closing_endp->DataPending();
	}

void TCP_Connection::UpdateEndpointVal(RecordVal* endp, int is_orig)
	{
	TCP_Endpoint* s = is_orig ? orig : resp;
	endp->Assign(0, new Val(s->Size(), TYPE_COUNT));
	endp->Assign(1, new Val(int(s->state), TYPE_COUNT));
	}

IMPLEMENT_SERIAL(TCP_Connection, SER_TCP_CONNECTION);

bool TCP_Connection::DoSerialize(SerialInfo* info) const
	{
	DO_SERIALIZE(SER_TCP_CONNECTION, Connection);

	SERIALIZE_OPTIONAL(analyzer);

	return SERIALIZE_BIT(is_partial) &&
		SERIALIZE_BIT(close_deferred) &&
		SERIALIZE_BIT(deferred_gen_event) &&
		orig->Serialize(info) &&
		resp->Serialize(info);
	}

bool TCP_Connection::DoUnserialize(UnserialInfo* info)
	{
	DO_UNSERIALIZE(Connection);

	UNSERIALIZE_OPTIONAL(analyzer, TCP_Analyzer::Unserialize(info));

	UNSERIALIZE_BIT(is_partial);
	UNSERIALIZE_BIT(close_deferred);
	UNSERIALIZE_BIT(deferred_gen_event);

	orig = TCP_Endpoint::Unserialize(info);
	if ( ! orig )
		return false;

	resp = TCP_Endpoint::Unserialize(info);
	if ( ! resp )
		return false;

	trace_rewriter = 0;
	src_pkt_writer = 0;

	return true;
	}

TCP_ConnectionContents::TCP_ConnectionContents(NetSessions* s,
		HashKey* k, double t, const ConnID* id,
		const struct tcphdr* tp, TCPContentsType arg_contents_type)
: TCP_Connection(s, k, t, id, tp)
	{
	contents_type = arg_contents_type;
	}

void TCP_ConnectionContents::BuildEndpoints()
	{
	if ( contents_type & TCP_CONTENTS_ORIG )
		orig->AddContentsProcessor(new TCP_Contents(orig));

	if ( contents_type & TCP_CONTENTS_RESP )
		resp->AddContentsProcessor(new TCP_Contents(resp));
	}

IMPLEMENT_SERIAL(TCP_ConnectionContents, SER_TCP_CONNECTION_CONTENTS);

bool TCP_ConnectionContents::DoSerialize(SerialInfo* info) const
	{
	DO_SERIALIZE(SER_TCP_CONNECTION_CONTENTS, TCP_Connection);
	return SERIALIZE(int(contents_type));
	}

bool TCP_ConnectionContents::DoUnserialize(UnserialInfo* info)
	{
	DO_UNSERIALIZE(TCP_Connection);
	int tmp;
	if ( ! UNSERIALIZE(&tmp) )
		return false;

	contents_type = (TCPContentsType) tmp;
	return true;
	}

unsigned int TCP_Connection::MemoryAllocation() const
	{
	// The following does detailed but expensive (when across all
	// connections) analysis of TCP data buffered by this connection,
	// either awaiting sequencing & processing, or awaiting acks from
	// the receiver.
#if 0
	int orig_waiting_on_hole, orig_waiting_on_ack;
	int resp_waiting_on_hole, resp_waiting_on_ack;

	orig->SizeBufferedData(orig_waiting_on_hole, orig_waiting_on_ack);
	resp->SizeBufferedData(resp_waiting_on_hole, resp_waiting_on_ack);

	if ( orig_waiting_on_hole + orig_waiting_on_ack + resp_waiting_on_hole + resp_waiting_on_ack > 0 )
		{
		printf("%.06f %d %d %d %d\n",
			start_time,
			orig_waiting_on_hole,
			orig_waiting_on_ack,
			resp_waiting_on_hole,
			resp_waiting_on_ack);
		}
#endif

	// FIXME: A rather low lower bound....
	return Connection::MemoryAllocation()
		+ padded_sizeof(*this) - padded_sizeof(Connection)
		+ (orig ? padded_sizeof(*orig) : 0)
		+ (resp ? padded_sizeof(*resp) : 0)
		+ (analyzer ? padded_sizeof(*analyzer) : 0)
		+ (trace_rewriter ? padded_sizeof(*trace_rewriter) : 0)
		+ (src_pkt_writer ? padded_sizeof(*src_pkt_writer) : 0);
	}

TCP_Analyzer::~TCP_Analyzer()
	{
	}

void TCP_Analyzer::Done()
	{
	done = 1;
	}

void TCP_Analyzer::Describe(ODesc* d) const
	{
	d->AddSP("analyzer for");
	conn->Describe(d);
	}

IMPLEMENT_SERIAL(TCP_Analyzer, SER_TCP_ANALYZER);

bool TCP_Analyzer::Serialize(SerialInfo* info) const
	{
	return SerialObj::Serialize(info);
	}

TCP_Analyzer* TCP_Analyzer::Unserialize(UnserialInfo* info)
	{
	return (TCP_Analyzer*) SerialObj::Unserialize(info, SER_TCP_ANALYZER);
	}

bool TCP_Analyzer::DoSerialize(SerialInfo* info) const
	{
	DO_SERIALIZE(SER_TCP_ANALYZER, BroObj);
	SERIALIZE_OPTIONAL(next);
	return conn->Serialize(info) && SERIALIZE(done);
	}

bool TCP_Analyzer::DoUnserialize(UnserialInfo* info)
	{
	DO_UNSERIALIZE(BroObj);

	UNSERIALIZE_OPTIONAL(next, TCP_Analyzer::Unserialize(info));
	conn = (TCP_Connection*) Connection::Unserialize(info);
	if ( ! conn )
		return false;

	return UNSERIALIZE(&done);
	}

TCP_Stats::TCP_Stats(TCP_Connection* c)
: TCP_Analyzer(c)
	{
	orig_stats = new TCP_EndpointStats(conn->Orig());
	resp_stats = new TCP_EndpointStats(conn->Resp());
	}

TCP_Stats::~TCP_Stats()
	{
	delete orig_stats;
	delete resp_stats;
	}

void TCP_Stats::Done()
	{
	val_list* vl = new val_list;
	vl->append(conn->BuildConnVal());
	vl->append(orig_stats->BuildStats());
	vl->append(resp_stats->BuildStats());

	conn->ConnectionEvent(conn_stats, vl);

	TCP_Analyzer::Done();
	}
