// $Id: TCP.h,v 1.7 2005/03/17 09:22:17 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef tcp_h
#define tcp_h

#include "Conn.h"
#include "NetVar.h"


class TCP_Connection;
class TCP_Contents;
class TCP_ContentLine;
class TCP_EndpointAnalyzer;
class TCP_Analyzer;

#include "TCP_Endpoint.h"
#include "TCP_Contents.h"

class TCP_Rewriter;
class TCP_SourcePacketWriter;
class TCP_Connection : public Connection {
public:
	TCP_Connection(NetSessions* s, HashKey* k, double t, const ConnID* id,
			const struct tcphdr* tp);
	virtual void Init();

	~TCP_Connection();
	void Done();

	void AddAnalyzer(TCP_Analyzer* a);
	void AnalyzerDone();

	int HasAnalyzers() const
		{ return orig->Analyzer() != 0 || resp->Analyzer() != 0; }

	void NextPacket(double t, int is_orig,
			const IP_Hdr* ip, int len, int caplen,
			const u_char*& data,
			int& record_packet, int& record_content,
			const struct pcap_pkthdr* hdr,
			const u_char* const pkt,
			int hdr_size);

	virtual void Deliver(TCP_Endpoint* sender, int seq, int len,
				u_char* data);
	virtual void Undelivered(TCP_Endpoint* sender, int seq, int len);

	// If this connection has a TCP_ContentLine, the following is called
	// for each new line of data.
	// const char*, and \0-terminated.  The caller should not modify
	// the data.  However, the default version of this method then calls
	// the next flavor, which is passed a char* instead.  The caller is
	// free to modify that data, but must not retain a pointer to it,
	// as it is volatile.
	//
	// const_len gives the number of const char's pointed to by line,
	// not including the newline.  If it's 0, that means that the
	// callee can safely cast line to char* and modify it (but not
	// keep a copy of a pointer to it, per the above).
	virtual void NewLine(TCP_ContentLine* sender, int length, const char* line);

	// "name" and "val" both now belong to this object, which needs to
	// delete them when done with them.
	virtual void SetEnv(int is_orig, char* name, char* val);

	void SetContentsFile(unsigned int direction, BroFile* f);
	BroFile* GetContentsFile(unsigned int direction) const;

	TCP_Rewriter* TraceRewriter() const	{ return trace_rewriter; }
	void SetTraceRewriter(TCP_Rewriter* rewriter);

	TCP_SourcePacketWriter* SourcePacketWriter() const
		{ return src_pkt_writer; }

	int IsReuse(double t, const u_char* pkt);

	// The given endpoint's data delivery is complete.
	virtual void EndpointEOF(TCP_Contents* endp);

	// TraceRewriterEOF is virtual so that individual protocols
	// can change the decision on whether to call EndOfData() or not.
	virtual void TraceRewriterEOF(TCP_Endpoint* endp);

	int IsPartial() const		{ return is_partial; }

	TCP_Endpoint* Orig() const	{ return orig; }
	TCP_Endpoint* Resp() const	{ return resp; }

	// True if the connection has closed in some sense, false otherwise.
	int IsClosed() const	{ return orig->did_close || resp->did_close; }
	int BothClosed() const	{ return orig->did_close && resp->did_close; }

	uint32 OrigSeq() const	{ return orig->last_seq; }
	uint32 RespSeq() const	{ return resp->last_seq; }

	int IsOrig(TCP_Endpoint* endp)	{ return endp == orig; }

	TransportProto ConnTransport() const	{ return TRANSPORT_TCP; }

	void Describe(ODesc* d) const;

	// Callback to process a TCP option.
	typedef int (*proc_tcp_option_t)(unsigned int opt, unsigned int optlen,
			const u_char* option, TCP_Connection* conn,
			int is_orig, void* cookie);

	// Needs to be static because it's passed as a pointer-to-function
	// rather than pointer-to-member-function.
	static int ParseTCPOptions(const struct tcphdr* tcp,
			proc_tcp_option_t proc, TCP_Connection* conn,
			int is_orig, void* cookie);

	virtual unsigned int MemoryAllocation() const;

	// If the connection compressor is activated, we need a special memory
	// layout. (See ConnCompressor.h).
	void* operator new(size_t size)
		{
		if ( ! use_connection_compressor )
			return ::operator new(size);

		void* c = ::operator new(size + 2);

		// We have to turn off the is_pending bit.  By setting the
		// first two bytes to zero, we'll achieve this.
		*((uint16*) c) = 0;

		return ((char *) c) + 2;
		}

	void operator delete(void* ptr)
		{
		if ( ! use_connection_compressor )
			::operator delete(ptr);
		else
			::operator delete(((char*) ptr) - 2);
		}

protected:
	TCP_Connection()	{ }

	virtual void BuildEndpoints();

	// True if either endpoint still has pending data.  closing_endp
	// is an endpoint that has indicated it is closing (i.e., for
	// which we have seen a FIN) - for it, data is pending unless
	// everything's been delivered up to the FIN.  For its peer,
	// the test is whether it has any outstanding, un-acked data.
	int DataPending(TCP_Endpoint* closing_endp);

	void UpdateEndpointVal(RecordVal* endp, int is_orig);

	// Called whenever an end enters TCP_CLOSED or TCP_RESET.  If
	// gen_event is true and the connection is now fully closed,
	// a connection_finished event will be generated; otherwise not.
	virtual void ConnectionClosed(TCP_Endpoint* endpoint,
					TCP_Endpoint* peer, int gen_event);
	virtual void ConnectionFinished(int half_finished);
	virtual void ConnectionReset();

	// Called whenever a RST packet is seen -- sometimes invocation
	// of ConnectionReset is delayed.
	virtual void PacketWithRST() { }

	friend class ConnectionTimer;
	void AttemptTimer(double t);
	void PartialCloseTimer(double t);
	void ExpireTimer(double t);
	void ResetTimer(double t);
	void DeleteTimer(double t);
	void ConnDeleteTimer(double t) { Connection::DeleteTimer(t); }

	Val* BuildSYNPacketVal(int is_orig,
				const IP_Hdr* ip, const struct tcphdr* tcp);

	RecordVal* BuildOSVal(int is_orig, const IP_Hdr* ip,
				const struct tcphdr* tcp, uint32 tcp_hdr_len);

	// Needs to be static because it's passed as a pointer-to-function
	// rather than pointer-to-member-function.
	static int TCPOptionEvent(unsigned int opt, unsigned int optlen,
				const u_char* option, TCP_Connection* conn,
				  int is_orig, void* cookie);

	DECLARE_SERIAL(TCP_Connection);

	TCP_Endpoint* orig;
	TCP_Endpoint* resp;

	TCP_Analyzer* analyzer;

	TCP_Rewriter* trace_rewriter;
	TCP_SourcePacketWriter* src_pkt_writer;

	unsigned int is_partial: 1;

	// Whether we're waiting on final data delivery before closing
	// this connection out.
	unsigned int close_deferred: 1;

	// Whether to generate an event when we finally do.
	unsigned int deferred_gen_event: 1;

};

typedef enum {
	// The numbers here are important, they're used for bit-masking.
	TCP_CONTENTS_NONE = 0,
	TCP_CONTENTS_ORIG = 1,	// reassemble originator stream
	TCP_CONTENTS_RESP = 2,	// reassemble responder stream
	TCP_CONTENTS_BOTH = 3,	// reassemble both sides
} TCPContentsType;

class TCP_ConnectionContents : public TCP_Connection {
public:
	TCP_ConnectionContents(NetSessions* s, HashKey* k, double t,
			const ConnID* id, const struct tcphdr* tp,
			TCPContentsType contents_type);

protected:
	TCP_ConnectionContents()	{ }

	void BuildEndpoints();
	DECLARE_SERIAL(TCP_ConnectionContents);

	TCPContentsType contents_type;
};

class TCP_Analyzer : public BroObj {
public:
	TCP_Analyzer(TCP_Connection* c)
		{
		conn = c;
		done = 0;
		next = 0;
		}

	virtual ~TCP_Analyzer();
	virtual void Done();

	// Add's 'a' to this analyzer's chain.  NOTE: this
	// analyzer is presumed to be a new one, so our "next"
	// field is currently empty.
	void AddAnalyzer(TCP_Analyzer* a)
		{
		if ( next )
			Internal("non-empty next field in AddAnalyzer");
		next = a;
		}

	TCP_Analyzer* NextAnalyzer()		{ return next; }

	TCP_Connection* Conn() const		{ return conn; }

	void Describe(ODesc* d) const;

	bool Serialize(SerialInfo* info) const;
	static TCP_Analyzer* Unserialize(UnserialInfo* info);

protected:
	TCP_Analyzer()	{}

	DECLARE_SERIAL(TCP_Analyzer)

	TCP_Analyzer* next;
	TCP_Connection* conn;
	int done;	// if true, corresponding connection has finished
};

class TCP_Stats : public TCP_Analyzer {
public:
	TCP_Stats(TCP_Connection* c);
	~TCP_Stats();
	void Done();

protected:
	TCP_EndpointStats* orig_stats;
	TCP_EndpointStats* resp_stats;
};


int TCP_Endpoint::IsOrig() const	{ return conn->Orig() == this; }

#endif
