// $Id: TCP_Contents.h,v 1.4 2004/11/28 23:27:54 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef tcpcontents_h
#define tcpcontents_h


#include "Reassem.h"

class BroFile;

class TCP_Reassembler : public Reassembler {
public:
	TCP_Reassembler(TCP_Contents* arg_contents_processor,
			const uint32* src_addr)
	: Reassembler(1, src_addr, REASSEM_TCP)
		{
		contents_processor = arg_contents_processor;
		record_contents_file = 0;
		}
	virtual ~TCP_Reassembler();

	// Returns the volume of data buffered in the reassembler.
	// First parameter returns data that is above a hole, and thus is
	// waiting on the hole being filled.  Second parameter returns
	// data that has been processed but is awaiting an ACK to free
	// it up.
	//
	// If we're not processing contents, then naturally each of
	// these is empty.
	void SizeBufferedData(int& waiting_on_hole, int& waiting_on_ack) const;

	// How much data is pending delivery since it's not yet reassembled.
	// Includes the data due to holes (so this value is a bit different
	// from waiting_on_hole above; and is computed in a different fashion).
	int NumUndeliveredBytes() const
		{
		if ( last_block )
			return last_block->upper - last_reassem_seq;
		else
			return 0;
		}

	void SetContentsFile(BroFile* f);
	BroFile* GetContentsFile() const	{ return record_contents_file; }

	void MatchUndelivered(int up_to_seq = -1);

protected:
	TCP_Reassembler()	{ }

	DECLARE_SERIAL(TCP_Reassembler);

	void Undelivered(int up_to_seq);

	void RecordToSeq(int start_seq, int stop_seq, BroFile* f);
	void RecordBlock(DataBlock* b, BroFile* f);
	void RecordGap(int start_seq, int upper_seq, BroFile* f);

	void BlockInserted(DataBlock* b);
	void Overlap(const u_char* b1, const u_char* b2, int n);

	TCP_Contents* contents_processor;
	BroFile* record_contents_file; // file on which to reassemble contents
};

class TCP_Contents : public SerialObj {
public:
	TCP_Contents(TCP_Endpoint* endp, int stop_on_gap = 0);
	virtual ~TCP_Contents();

	int DataSent(double t, int seq, int len, const u_char* data);
	void AckReceived(int seq);

	// CheckEOF checks if we have delivered all contents that we
	// can possibly deliver for this endpoint.  It calls
	// TCP_Connection::EndpointEOF() when so.
	void CheckEOF();

	int HasUndeliveredData() const	{ return t_reassem->HasBlocks(); }
	int DataPending() const;
	void MatchUndeliveredData()	{ t_reassem->MatchUndelivered(); }
	int DataSeq() const		{ return t_reassem->LastReassemSeq(); }

	void DeliverBlock(int seq, int len, u_char* data);
	virtual void Deliver(int seq, int len, u_char* data);
	virtual void Undelivered(int seq, int len);

	void SizeBufferedData(int& waiting_on_hole, int& waiting_on_ack);

	int SkippingDeliveries() const	{ return skip_deliveries; }
	void SetSkipDeliveries(int should_skip)
		{ skip_deliveries = should_skip; }

	void SetContentsFile(BroFile* f);
	BroFile* GetContentsFile() const;

	TCP_Endpoint* Endpoint()	{ return endp; }
	const TCP_Endpoint* Endpoint() const	{ return endp; }

	TCP_Connection* Conn()	{ return endp->Conn(); }
	const TCP_Connection* Conn() const	{ return endp->Conn(); }

	int IsOrig() const	{ return endp->IsOrig(); }

	bool Serialize(SerialInfo* info) const;
	static TCP_Contents* Unserialize(UnserialInfo* info);

protected:
	TCP_Contents()	{}

	DECLARE_SERIAL(TCP_Contents)

	TCP_Endpoint* endp;
	TCP_Reassembler* t_reassem;
	unsigned int deliver_tcp_contents:1;
	unsigned int skip_deliveries:1;	// if true, skip future data delivery
	unsigned int stop_on_gap:1;
	unsigned int did_EOF:1;
};

#define CR_as_EOL 1
#define LF_as_EOL 2

class TCP_ContentLine : public TCP_Contents {
public:
	TCP_ContentLine(TCP_Endpoint* endp,
			int is_NUL_sensitive, int skip_partial,
			int CRLF_as_EOL = (CR_as_EOL | LF_as_EOL));
	~TCP_ContentLine();

	int HasPartialLine() const	{ return buf && offset > 0; }

protected:
	TCP_ContentLine()	{ }

	void Init(int size);
	void Deliver(int seq, int len, u_char* data);
	virtual void DoDeliver(int seq, int len, u_char* data);
	virtual int DoDeliverOnce(int len, u_char* data);
	virtual void ExcessiveLine(const u_char* data, int len);
	u_char* NextNewLine(int len, u_char* data, int& offset);
	void CheckNUL();

	DECLARE_SERIAL(TCP_ContentLine);

	char* buf;	// where we build up the body of the request
	int offset;	// where we are in buf
	int buf_len;	// how big buf is, total
	unsigned int last_char;	// last (non-option) character scanned

	// If true, flag (first) line with embedded NUL.
	unsigned int flag_NULs:1;

	// If true, don't process TCP_PARTIAL conn's.
	unsigned int punt_on_partial:1;

	// If non-zero, flag lines longer than this.
	unsigned int line_limit:1;

	// Whether single CR / LF are considered as EOL.
	unsigned int CR_LF_as_EOL:2;
};

#endif
