// $Id: TCP_Endpoint.h,v 1.6 2004/11/28 23:27:54 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef tcpendpoint_h
#define tcpendpoint_h


typedef enum {
	TCP_INACTIVE,	// no SYN (or any other packets) seen for this side
	TCP_SYN_SENT,	// SYN seen, but no ack
	TCP_SYN_ACK_SENT,	// SYN ack seen, no initial SYN
	TCP_PARTIAL,	// data seen, but no SYN
	TCP_ESTABLISHED,	// SYN ack seen (implicit for SYN sent by
				// responder)
	TCP_CLOSED,	// FIN seen
	TCP_RESET	// RST seen
} EndpointState;

class TCP_Connection;
class TCP_Contents;
class TCP_EndpointAnalyzer;
class IP_Hdr;

// One endpoint of a TCP connection.
class TCP_Endpoint : public SerialObj {
public:
	TCP_Endpoint(TCP_Connection* conn, int is_orig);
	~TCP_Endpoint();

	void SetPeer(TCP_Endpoint* p);

	void SetState(EndpointState new_state);
	int Size() const;
	int IsActive() const	{ return state != TCP_INACTIVE && ! did_close; }

	double StartTime() const	{ return start_time; }
	double LastTime() const		{ return last_time; }

	uint32 StartSeq() const		{ return start_seq; }
	uint32 LastSeq() const		{ return last_seq; }
	uint32 AckSeq() const		{ return ack_seq; }

	// True if none of this endpoint's data has been acknowledged.
	// We allow for possibly one octet being ack'd in the case of
	// an initial SYN exchange.
	int NoDataAcked() const
		{ return ack_seq == start_seq || ack_seq == start_seq + 1; }

	TCP_Connection* Conn() const	{ return conn; }

	int HasContents() const		{ return contents_processor != 0; }

	inline int IsOrig() const;

	int HasDoneSomething() const	{ return last_time != 0.0; }

	void AddContentsProcessor(TCP_Contents* contents_processor);

	TCP_EndpointAnalyzer* Analyzer()		{ return analyzer; }
	void AddAnalyzer(TCP_EndpointAnalyzer* a);

	int DataPending() const;
	int HasUndeliveredData() const;
	void MatchUndeliveredData();
	void CheckEOF();

	// Returns the volume of data buffered in the reassembler.
	// First parameter returns data that is above a hole, and thus is
	// waiting on the hole being filled.  Second parameter returns
	// data that has been processed but is awaiting an ACK to free
	// it up.
	//
	// If we're not processing contents, then naturally each of
	// these is empty.
	void SizeBufferedData(int& waiting_on_hole, int& waiting_on_ack);

	int ValidChecksum(const struct tcphdr* tp, int len) const;

	// Returns true if the data was used (and hence should be recorded
	// in the save file), false otherwise.
	int DataSent(double t, int seq, int len, const u_char* data,
			const IP_Hdr* ip, const struct tcphdr* tp);

	void AckReceived(int seq);

	void SetContentsFile(BroFile* f);
	BroFile* GetContentsFile() const	{ return contents_file; }

	bool Serialize(SerialInfo* info) const;
	static TCP_Endpoint* Unserialize(UnserialInfo* info);

	//### combine into a set of flags:
	EndpointState state, prev_state;
	TCP_Endpoint* peer;
	TCP_Contents* contents_processor;
	TCP_EndpointAnalyzer* analyzer;
	TCP_Connection* conn;
	BroFile* contents_file;
	uint32 checksum_base;

	double start_time, last_time;
	uint32 start_seq, last_seq, ack_seq;	// in host order
	const uint32* src_addr; // the other endpoint
	const uint32* dst_addr; // this endpoint
	uint32 window; // current congestion window (*scaled*, not pre-scaling)
	int window_scale;  // from the TCP option
	uint32 window_ack_seq; // at which ack_seq number did we record 'window'
	uint32 window_seq; // at which sending sequence number did we record 'window'
	int contents_start_seq;	// relative seq # where contents file starts
	int FIN_seq;		// relative seq # to start_seq
	int SYN_cnt, FIN_cnt, RST_cnt;
	int did_close;		// whether we've reported it closing

protected:
	TCP_Endpoint()	{}

	DECLARE_SERIAL(TCP_Endpoint)
};

class TCP_EndpointAnalyzer: public BroObj {
public:
	TCP_EndpointAnalyzer(TCP_Endpoint* e);
	virtual ~TCP_EndpointAnalyzer();

	// Add's 'a' to this analyzer's chain.  NOTE: this
	// analyzer is presumed to be a new one, so our "next"
	// field is currently empty.
	void AddAnalyzer(TCP_EndpointAnalyzer* a)
		{
		if ( next )
			Internal("non-empty next field in AddAnalyzer");
		next = a;
		}

	virtual int DataSent(double t, int seq, int len, const u_char* data,
				const IP_Hdr* ip, const struct tcphdr* tp);

	TCP_EndpointAnalyzer* NextAnalyzer()	{ return next; }

	TCP_Endpoint* Endpoint() const	{ return endp; }
	TCP_Connection* Conn() const	{ return endp->Conn(); }

	void Describe(ODesc* d) const;

	bool Serialize(SerialInfo* info) const;
	static TCP_EndpointAnalyzer* Unserialize(UnserialInfo* info);

protected:
	TCP_EndpointAnalyzer()	{}

	DECLARE_SERIAL(TCP_EndpointAnalyzer)

	TCP_Endpoint* endp;
	TCP_EndpointAnalyzer* next;
};


#define ENDIAN_UNKNOWN 0
#define ENDIAN_LITTLE 1
#define ENDIAN_BIG 2
#define ENDIAN_CONFUSED 3

class TCP_EndpointStats : public TCP_EndpointAnalyzer {
public:
	TCP_EndpointStats(TCP_Endpoint* endp);

	int DataSent(double t, int seq, int len, const u_char* data,
			const IP_Hdr* ip, const struct tcphdr* tp);

	RecordVal* BuildStats();

protected:
	int num_pkts;
	int num_rxmit;
	int num_rxmit_bytes;
	int num_in_order;
	int num_OO;
	int num_repl;
	int max_top_seq;
	int last_id;
	int endian_type;
};


#endif
