// $Id: Timer.h,v 1.5 2005/03/08 15:30:24 vern Exp $
//
// Copyright (c) 1996, 1997, 1998, 1999, 2001, 2002
//      The Regents of the University of California.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that: (1) source code distributions
// retain the above copyright notice and this paragraph in its entirety, (2)
// distributions including binary code include the above copyright notice and
// this paragraph in its entirety in the documentation or other materials
// provided with the distribution, and (3) all advertising materials mentioning
// features or use of this software display the following acknowledgement:
// ``This product includes software developed by the University of California,
// Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
// the University nor the names of its contributors may be used to endorse
// or promote products derived from this software without specific prior
// written permission.
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
// WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef timer_h
#define timer_h

#include "SerialObj.h"
#include "PriorityQueue.h"

extern "C" {
#include "cq.h"
}

// If you add a timer here, adjust TimerNames in Timer.cc.
enum TimerType {
	TIMER_BACKDOOR,
	TIMER_BREAKPOINT,
	TIMER_CONN_DELETE,
	TIMER_CONN_EXPIRE,
	TIMER_CONN_INACTIVITY,
	TIMER_CONN_STATUS_UPDATE,
	TIMER_DNS_EXPIRE,
	TIMER_FRAG,
	TIMER_INCREMENTAL_WRITE,
	TIMER_INTERCONN,
	TIMER_NB_EXPIRE,
	TIMER_NETWORK,
	TIMER_NTP_EXPIRE,
	TIMER_PROFILE,
	TIMER_ROTATE,
	TIMER_RPC_EXPIRE,
	TIMER_SCHEDULE,
	TIMER_TABLE_VAL,
	TIMER_TCP_ATTEMPT,
	TIMER_TCP_DELETE,
	TIMER_TCP_EXPIRE,
	TIMER_TCP_PARTIAL_CLOSE,
	TIMER_TCP_RESET,
};
const int NUM_TIMER_TYPES = int(TIMER_TCP_RESET) + 1;

extern const char* timer_type_to_string(TimerType type);

class Serializer;
class ODesc;

class Timer : public SerialObj, public PQ_Element {
public:
	Timer(double t, TimerType arg_type) : PQ_Element(t)
		{ type = (char) arg_type; }
	virtual ~Timer()	{ }

	TimerType Type() const	{ return (TimerType) type; }

	// t gives the dispatch time.  is_expire is true if the
	// timer is being dispatched because we're expiring all
	// pending timers.
	virtual void Dispatch(double t, int is_expire) = 0;

	void Describe(ODesc* d) const;

	bool Serialize(SerialInfo* info) const;
	static Timer* Unserialize(UnserialInfo* info);

protected:
	Timer()	{}

	DECLARE_ABSTRACT_SERIAL(Timer);

	unsigned int type:8;
};

class TimerMgr {
public:
	virtual ~TimerMgr();

	virtual void Add(Timer* timer) = 0;

	// Advance the clock to time t, expiring at most max_expire timers.
	// Returns number of timers expired.
	int Advance(double t, int max_expire);

	// Returns the number of timers expired (so far) during the current
	// or most recent advance.
	int NumExpiredDuringCurrentAdvance()	{ return num_expired; }

	// Expire all timers.
	virtual void Expire() = 0;

	// Cancel() is a method separate from Remove because
	// (1) Remove is protected, but, more importantly, (2) in some
	// timer schemes we have wound up separating timer cancelation
	// from removing it from the manager's data structures, because
	// the manager lacked an efficient way to find it.
	void Cancel(Timer* timer)	{ Remove(timer); }

	double Time() const		{ return t; }

	virtual int Size() const = 0;
	virtual int PeakSize() const = 0;

	double LastTimestamp() const	{ return last_timestamp; }
	
	static unsigned int* CurrentTimers()	{ return current_timers; }

protected:
	TimerMgr()	{ t = last_timestamp = 0.0; num_expired = 0; }

	virtual int DoAdvance(double t, int max_expire) = 0;
	virtual void Remove(Timer* timer) = 0;

	double t;
	double last_timestamp;

	int num_expired;

	static unsigned int current_timers[NUM_TIMER_TYPES];
};

class PQ_TimerMgr : public TimerMgr {
public:
	PQ_TimerMgr();
	~PQ_TimerMgr();

	void Add(Timer* timer);
	void Expire();

	int Size() const	{ return q->Size(); }
	int PeakSize() const	{ return q->PeakSize(); }
	unsigned int MemoryUsage() const;

protected:
	int DoAdvance(double t, int max_expire);
	void Remove(Timer* timer);

	Timer* Remove()			{ return (Timer*) q->Remove(); }
	Timer* Top()			{ return (Timer*) q->Top(); }

	PriorityQueue* q;
};

class CQ_TimerMgr : public TimerMgr {
public:
	CQ_TimerMgr();
	~CQ_TimerMgr();

	void Add(Timer* timer);
	void Expire();

	int Size() const	{ return cq_size(cq); }
	int PeakSize() const	{ return cq_max_size(cq); }
	unsigned int MemoryUsage() const;

protected:
	int DoAdvance(double t, int max_expire);
	void Remove(Timer* timer);

	struct cq_handle *cq;
};

extern TimerMgr* timer_mgr;

#endif
