module SSL;

export {

	const SSLv2  = 0x0002;
	const SSLv3  = 0x0300;
	const TLSv10 = 0x0301;
	const TLSv11 = 0x0302;
	const version_strings: table[count] of string = {
		[SSLv2] = "SSLv2",
		[SSLv3] = "SSLv3",
		[TLSv10] = "TLSv10",
		[TLSv11] = "TLSv11",
	} &default="UNKNOWN";

	# http://www.iana.org/assignments/tls-extensiontype-values/tls-extensiontype-values.xml
	const extensions: table[count] of string = {
		[0] = "server_name",
		[1] = "max_fragment_length",
		[2] = "client_certificate_url",
		[3] = "trusted_ca_keys",
		[4] = "truncated_hmac",
		[5] = "status_request",
		[6] = "user_mapping",
		[7] = "client_authz",
		[8] = "server_authz",
		[9] = "cert_type",
		[10] = "elliptic_curves",
		[11] = "ec_point_formats",
		[12] = "srp",
		[13] = "signature_algorithms",
		[14] = "use_srtp",
		[35] = "SessionTicket TLS",
		[65281] = "renegotiation_info"
	} &default=function(i: count):string { return fmt("unknown-%d", i); };
	
	## SSLv2
	const SSLv20_CK_RC4_128_WITH_MD5 = 0x010080;
	const SSLv20_CK_RC4_128_EXPORT40_WITH_MD5 = 0x020080;
	const SSLv20_CK_RC2_128_CBC_WITH_MD5 = 0x030080;
	const SSLv20_CK_RC2_128_CBC_EXPORT40_WITH_MD5 = 0x040080;
	const SSLv20_CK_IDEA_128_CBC_WITH_MD5 = 0x050080;
	const SSLv20_CK_DES_64_CBC_WITH_MD5 = 0x060040;
	const SSLv20_CK_DES_192_EDE3_CBC_WITH_MD5 = 0x0700C0;

	## TLS
	const TLS_NULL_WITH_NULL_NULL = 0x0000;
	const TLS_RSA_WITH_NULL_MD5 = 0x0001;
	const TLS_RSA_WITH_NULL_SHA = 0x0002;
	const TLS_RSA_EXPORT_WITH_RC4_40_MD5 = 0x0003;
	const TLS_RSA_WITH_RC4_128_MD5 = 0x0004;
	const TLS_RSA_WITH_RC4_128_SHA = 0x0005;
	const TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5 = 0x0006;
	const TLS_RSA_WITH_IDEA_CBC_SHA = 0x0007;
	const TLS_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x0008;
	const TLS_RSA_WITH_DES_CBC_SHA = 0x0009;
	const TLS_RSA_WITH_3DES_EDE_CBC_SHA = 0x000A;
	const TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA = 0x000B;
	const TLS_DH_DSS_WITH_DES_CBC_SHA = 0x000C;
	const TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA = 0x000D;
	const TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x000E;
	const TLS_DH_RSA_WITH_DES_CBC_SHA = 0x000F;
	const TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA = 0x0010;
	const TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA = 0x0011;
	const TLS_DHE_DSS_WITH_DES_CBC_SHA = 0x0012;
	const TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA = 0x0013;
	const TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x0014;
	const TLS_DHE_RSA_WITH_DES_CBC_SHA = 0x0015;
	const TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA = 0x0016;
	const TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5 = 0x0017;
	const TLS_DH_ANON_WITH_RC4_128_MD5 = 0x0018;
	const TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA = 0x0019;
	const TLS_DH_ANON_WITH_DES_CBC_SHA = 0x001A;
	const TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA = 0x001B;
	const SSL_FORTEZZA_KEA_WITH_NULL_SHA = 0x001C;
	const SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA = 0x001D;
	const TLS_KRB5_WITH_DES_CBC_SHA = 0x001E;
	const TLS_KRB5_WITH_3DES_EDE_CBC_SHA = 0x001F;
	const TLS_KRB5_WITH_RC4_128_SHA = 0x0020;
	const TLS_KRB5_WITH_IDEA_CBC_SHA = 0x0021;
	const TLS_KRB5_WITH_DES_CBC_MD5 = 0x0022;
	const TLS_KRB5_WITH_3DES_EDE_CBC_MD5 = 0x0023;
	const TLS_KRB5_WITH_RC4_128_MD5 = 0x0024;
	const TLS_KRB5_WITH_IDEA_CBC_MD5 = 0x0025;
	const TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA = 0x0026;
	const TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA = 0x0027;
	const TLS_KRB5_EXPORT_WITH_RC4_40_SHA = 0x0028;
	const TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5 = 0x0029;
	const TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5 = 0x002A;
	const TLS_KRB5_EXPORT_WITH_RC4_40_MD5 = 0x002B;
	const TLS_RSA_WITH_AES_128_CBC_SHA = 0x002F;
	const TLS_DH_DSS_WITH_AES_128_CBC_SHA = 0x0030;
	const TLS_DH_RSA_WITH_AES_128_CBC_SHA = 0x0031;
	const TLS_DHE_DSS_WITH_AES_128_CBC_SHA = 0x0032;
	const TLS_DHE_RSA_WITH_AES_128_CBC_SHA = 0x0033;
	const TLS_DH_ANON_WITH_AES_128_CBC_SHA = 0x0034;
	const TLS_RSA_WITH_AES_256_CBC_SHA = 0x0035;
	const TLS_DH_DSS_WITH_AES_256_CBC_SHA = 0x0036;
	const TLS_DH_RSA_WITH_AES_256_CBC_SHA = 0x0037;
	const TLS_DHE_DSS_WITH_AES_256_CBC_SHA = 0x0038;
	const TLS_DHE_RSA_WITH_AES_256_CBC_SHA = 0x0039;
	const TLS_DH_ANON_WITH_AES_256_CBC_SHA = 0x003A;
	const TLS_RSA_WITH_NULL_SHA256 = 0x003B;
	const TLS_RSA_WITH_AES_128_CBC_SHA256 = 0x003C;
	const TLS_RSA_WITH_AES_256_CBC_SHA256 = 0x003D;
	const TLS_DH_DSS_WITH_AES_128_CBC_SHA256 = 0x003E;
	const TLS_DH_RSA_WITH_AES_128_CBC_SHA256 = 0x003F;
	const TLS_DHE_DSS_WITH_AES_128_CBC_SHA256 = 0x0040;
	const TLS_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0041;
	const TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA = 0x0042;
	const TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0043;
	const TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA = 0x0044;
	const TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0045;
	const TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA = 0x0046;
	const TLS_RSA_EXPORT1024_WITH_RC4_56_MD5 = 0x0060;
	const TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5 = 0x0061;
	const TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA = 0x0062;
	const TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA = 0x0063;
	const TLS_RSA_EXPORT1024_WITH_RC4_56_SHA = 0x0064;
	const TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA = 0x0065;
	const TLS_DHE_DSS_WITH_RC4_128_SHA = 0x0066;
	const TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 = 0x0067;
	const TLS_DH_DSS_WITH_AES_256_CBC_SHA256 = 0x0068;
	const TLS_DH_RSA_WITH_AES_256_CBC_SHA256 = 0x0069;
	const TLS_DHE_DSS_WITH_AES_256_CBC_SHA256 = 0x006A;
	const TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 = 0x006B;
	const TLS_DH_ANON_WITH_AES_128_CBC_SHA256 = 0x006C;
	const TLS_DH_ANON_WITH_AES_256_CBC_SHA256 = 0x006D;
	const TLS_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0084;
	const TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA = 0x0085;
	const TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0086;
	const TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA = 0x0087;
	const TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0088;
	const TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA = 0x0089;
	const TLS_PSK_WITH_RC4_128_SHA = 0x008A;
	const TLS_PSK_WITH_3DES_EDE_CBC_SHA = 0x008B;
	const TLS_PSK_WITH_AES_128_CBC_SHA = 0x008C;
	const TLS_PSK_WITH_AES_256_CBC_SHA = 0x008D;
	const TLS_DHE_PSK_WITH_RC4_128_SHA = 0x008E;
	const TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA = 0x008F;
	const TLS_DHE_PSK_WITH_AES_128_CBC_SHA = 0x0090;
	const TLS_DHE_PSK_WITH_AES_256_CBC_SHA = 0x0091;
	const TLS_RSA_PSK_WITH_RC4_128_SHA = 0x0092;
	const TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA = 0x0093;
	const TLS_RSA_PSK_WITH_AES_128_CBC_SHA = 0x0094;
	const TLS_RSA_PSK_WITH_AES_256_CBC_SHA = 0x0095;
	const TLS_RSA_WITH_SEED_CBC_SHA = 0x0096;
	const TLS_DH_DSS_WITH_SEED_CBC_SHA = 0x0097;
	const TLS_DH_RSA_WITH_SEED_CBC_SHA = 0x0098;
	const TLS_DHE_DSS_WITH_SEED_CBC_SHA = 0x0099;
	const TLS_DHE_RSA_WITH_SEED_CBC_SHA = 0x009A;
	const TLS_DH_ANON_WITH_SEED_CBC_SHA = 0x009B;
	const TLS_RSA_WITH_AES_128_GCM_SHA256 = 0x009C;
	const TLS_RSA_WITH_AES_256_GCM_SHA384 = 0x009D;
	const TLS_DHE_RSA_WITH_AES_128_GCM_SHA256 = 0x009E;
	const TLS_DHE_RSA_WITH_AES_256_GCM_SHA384 = 0x009F;
	const TLS_DH_RSA_WITH_AES_128_GCM_SHA256 = 0x00A0;
	const TLS_DH_RSA_WITH_AES_256_GCM_SHA384 = 0x00A1;
	const TLS_DHE_DSS_WITH_AES_128_GCM_SHA256 = 0x00A2;
	const TLS_DHE_DSS_WITH_AES_256_GCM_SHA384 = 0x00A3;
	const TLS_DH_DSS_WITH_AES_128_GCM_SHA256 = 0x00A4;
	const TLS_DH_DSS_WITH_AES_256_GCM_SHA384 = 0x00A5;
	const TLS_DH_ANON_WITH_AES_128_GCM_SHA256 = 0x00A6;
	const TLS_DH_ANON_WITH_AES_256_GCM_SHA384 = 0x00A7;
	const TLS_PSK_WITH_AES_128_GCM_SHA256 = 0x00A8;
	const TLS_PSK_WITH_AES_256_GCM_SHA384 = 0x00A9;
	const TLS_DHE_PSK_WITH_AES_128_GCM_SHA256 = 0x00AA;
	const TLS_DHE_PSK_WITH_AES_256_GCM_SHA384 = 0x00AB;
	const TLS_RSA_PSK_WITH_AES_128_GCM_SHA256 = 0x00AC;
	const TLS_RSA_PSK_WITH_AES_256_GCM_SHA384 = 0x00AD;
	const TLS_PSK_WITH_AES_128_CBC_SHA256 = 0x00AE;
	const TLS_PSK_WITH_AES_256_CBC_SHA384 = 0x00AF;
	const TLS_PSK_WITH_NULL_SHA256 = 0x00B0;
	const TLS_PSK_WITH_NULL_SHA384 = 0x00B1;
	const TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 = 0x00B2;
	const TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 = 0x00B3;
	const TLS_DHE_PSK_WITH_NULL_SHA256 = 0x00B4;
	const TLS_DHE_PSK_WITH_NULL_SHA384 = 0x00B5;
	const TLS_RSA_PSK_WITH_AES_128_CBC_SHA256 = 0x00B6;
	const TLS_RSA_PSK_WITH_AES_256_CBC_SHA384 = 0x00B7;
	const TLS_RSA_PSK_WITH_NULL_SHA256 = 0x00B8;
	const TLS_RSA_PSK_WITH_NULL_SHA384 = 0x00B9;
	const TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BA;
	const TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BB;
	const TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BC;
	const TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BD;
	const TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BE;
	const TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BF;
	const TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C0;
	const TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C1;
	const TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C2;
	const TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C3;
	const TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C4;
	const TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C5;
	const TLS_ECDH_ECDSA_WITH_NULL_SHA = 0xC001;
	const TLS_ECDH_ECDSA_WITH_RC4_128_SHA = 0xC002;
	const TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA = 0xC003;
	const TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA = 0xC004;
	const TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA = 0xC005;
	const TLS_ECDHE_ECDSA_WITH_NULL_SHA = 0xC006;
	const TLS_ECDHE_ECDSA_WITH_RC4_128_SHA = 0xC007;
	const TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA = 0xC008;
	const TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA = 0xC009;
	const TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA = 0xC00A;
	const TLS_ECDH_RSA_WITH_NULL_SHA = 0xC00B;
	const TLS_ECDH_RSA_WITH_RC4_128_SHA = 0xC00C;
	const TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA = 0xC00D;
	const TLS_ECDH_RSA_WITH_AES_128_CBC_SHA = 0xC00E;
	const TLS_ECDH_RSA_WITH_AES_256_CBC_SHA = 0xC00F;
	const TLS_ECDHE_RSA_WITH_NULL_SHA = 0xC010;
	const TLS_ECDHE_RSA_WITH_RC4_128_SHA = 0xC011;
	const TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA = 0xC012;
	const TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA = 0xC013;
	const TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA = 0xC014;
	const TLS_ECDH_ANON_WITH_NULL_SHA = 0xC015;
	const TLS_ECDH_ANON_WITH_RC4_128_SHA = 0xC016;
	const TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA = 0xC017;
	const TLS_ECDH_ANON_WITH_AES_128_CBC_SHA = 0xC018;
	const TLS_ECDH_ANON_WITH_AES_256_CBC_SHA = 0xC019;
	const TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA = 0xC01A;
	const TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA = 0xC01B;
	const TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA = 0xC01C;
	const TLS_SRP_SHA_WITH_AES_128_CBC_SHA = 0xC01D;
	const TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA = 0xC01E;
	const TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA = 0xC01F;
	const TLS_SRP_SHA_WITH_AES_256_CBC_SHA = 0xC020;
	const TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA = 0xC021;
	const TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA = 0xC022;
	const TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256 = 0xC023;
	const TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384 = 0xC024;
	const TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 = 0xC025;
	const TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 = 0xC026;
	const TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256 = 0xC027;
	const TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384 = 0xC028;
	const TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256 = 0xC029;
	const TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384 = 0xC02A;
	const TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 = 0xC02B;
	const TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 = 0xC02C;
	const TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256 = 0xC02D;
	const TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384 = 0xC02E;
	const TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256 = 0xC02F;
	const TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384 = 0xC030;
	const TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256 = 0xC031;
	const TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384 = 0xC032;
	const TLS_ECDHE_PSK_WITH_RC4_128_SHA = 0xC033;
	const TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA = 0xC034;
	const TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA = 0xC035;
	const TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA = 0xC036;
	const TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256 = 0xC037;
	const TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384 = 0xC038;
	const TLS_ECDHE_PSK_WITH_NULL_SHA = 0xC039;
	const TLS_ECDHE_PSK_WITH_NULL_SHA256 = 0xC03A;
	const TLS_ECDHE_PSK_WITH_NULL_SHA384 = 0xC03B;
	const SSL_RSA_FIPS_WITH_DES_CBC_SHA = 0xFEFE;
	const SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA = 0xFEFF;
	const SSL_RSA_FIPS_WITH_DES_CBC_SHA_2 = 0xFFE1;
	const SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA_2 = 0xFFE0;
	const SSL_RSA_WITH_RC2_CBC_MD5 = 0xFF80;
	const SSL_RSA_WITH_IDEA_CBC_MD5 = 0xFF81;
	const SSL_RSA_WITH_DES_CBC_MD5 = 0xFF82;
	const SSL_RSA_WITH_3DES_EDE_CBC_MD5 = 0xFF83;
	const TLS_EMPTY_RENEGOTIATION_INFO_SCSV = 0x00FF;

	# --- This is a table of all known cipher specs.
	# --- It can be used for detecting unknown ciphers and for
	# --- converting the cipher spec constants into a human readable format.

	const cipher_desc: table[count] of string = {
		# --- sslv20 ---
		[SSLv20_CK_RC4_128_EXPORT40_WITH_MD5] =
			"SSLv20_CK_RC4_128_EXPORT40_WITH_MD5",
		[SSLv20_CK_RC4_128_WITH_MD5] = "SSLv20_CK_RC4_128_WITH_MD5",
		[SSLv20_CK_RC2_128_CBC_WITH_MD5] = "SSLv20_CK_RC2_128_CBC_WITH_MD5",
		[SSLv20_CK_RC2_128_CBC_EXPORT40_WITH_MD5] =
			"SSLv20_CK_RC2_128_CBC_EXPORT40_WITH_MD5",
		[SSLv20_CK_IDEA_128_CBC_WITH_MD5] = "SSLv20_CK_IDEA_128_CBC_WITH_MD5",
		[SSLv20_CK_DES_192_EDE3_CBC_WITH_MD5] =
			"SSLv20_CK_DES_192_EDE3_CBC_WITH_MD5",
		[SSLv20_CK_DES_64_CBC_WITH_MD5] = "SSLv20_CK_DES_64_CBC_WITH_MD5",

		# --- TLS ---
		[TLS_NULL_WITH_NULL_NULL] = "TLS_NULL_WITH_NULL_NULL",
		[TLS_RSA_WITH_NULL_MD5] = "TLS_RSA_WITH_NULL_MD5",
		[TLS_RSA_WITH_NULL_SHA] = "TLS_RSA_WITH_NULL_SHA",
		[TLS_RSA_EXPORT_WITH_RC4_40_MD5] = "TLS_RSA_EXPORT_WITH_RC4_40_MD5",
		[TLS_RSA_WITH_RC4_128_MD5] = "TLS_RSA_WITH_RC4_128_MD5",
		[TLS_RSA_WITH_RC4_128_SHA] = "TLS_RSA_WITH_RC4_128_SHA",
		[TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5] = "TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5",
		[TLS_RSA_WITH_IDEA_CBC_SHA] = "TLS_RSA_WITH_IDEA_CBC_SHA",
		[TLS_RSA_EXPORT_WITH_DES40_CBC_SHA] = "TLS_RSA_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_RSA_WITH_DES_CBC_SHA] = "TLS_RSA_WITH_DES_CBC_SHA",
		[TLS_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DH_DSS_WITH_DES_CBC_SHA] = "TLS_DH_DSS_WITH_DES_CBC_SHA",
		[TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA] = "TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA",
		[TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DH_RSA_WITH_DES_CBC_SHA] = "TLS_DH_RSA_WITH_DES_CBC_SHA",
		[TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DHE_DSS_WITH_DES_CBC_SHA] = "TLS_DHE_DSS_WITH_DES_CBC_SHA",
		[TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA] = "TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA",
		[TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DHE_RSA_WITH_DES_CBC_SHA] = "TLS_DHE_RSA_WITH_DES_CBC_SHA",
		[TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5] = "TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5",
		[TLS_DH_ANON_WITH_RC4_128_MD5] = "TLS_DH_ANON_WITH_RC4_128_MD5",
		[TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DH_ANON_WITH_DES_CBC_SHA] = "TLS_DH_ANON_WITH_DES_CBC_SHA",
		[TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA] = "TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA",
		[SSL_FORTEZZA_KEA_WITH_NULL_SHA] = "SSL_FORTEZZA_KEA_WITH_NULL_SHA",
		[SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA] = "SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA",
		[TLS_KRB5_WITH_DES_CBC_SHA] = "TLS_KRB5_WITH_DES_CBC_SHA",
		[TLS_KRB5_WITH_3DES_EDE_CBC_SHA] = "TLS_KRB5_WITH_3DES_EDE_CBC_SHA",
		[TLS_KRB5_WITH_RC4_128_SHA] = "TLS_KRB5_WITH_RC4_128_SHA",
		[TLS_KRB5_WITH_IDEA_CBC_SHA] = "TLS_KRB5_WITH_IDEA_CBC_SHA",
		[TLS_KRB5_WITH_DES_CBC_MD5] = "TLS_KRB5_WITH_DES_CBC_MD5",
		[TLS_KRB5_WITH_3DES_EDE_CBC_MD5] = "TLS_KRB5_WITH_3DES_EDE_CBC_MD5",
		[TLS_KRB5_WITH_RC4_128_MD5] = "TLS_KRB5_WITH_RC4_128_MD5",
		[TLS_KRB5_WITH_IDEA_CBC_MD5] = "TLS_KRB5_WITH_IDEA_CBC_MD5",
		[TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA] = "TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA",
		[TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA] = "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA",
		[TLS_KRB5_EXPORT_WITH_RC4_40_SHA] = "TLS_KRB5_EXPORT_WITH_RC4_40_SHA",
		[TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5] = "TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5",
		[TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5] = "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5",
		[TLS_KRB5_EXPORT_WITH_RC4_40_MD5] = "TLS_KRB5_EXPORT_WITH_RC4_40_MD5",
		[TLS_RSA_WITH_AES_128_CBC_SHA] = "TLS_RSA_WITH_AES_128_CBC_SHA",
		[TLS_DH_DSS_WITH_AES_128_CBC_SHA] = "TLS_DH_DSS_WITH_AES_128_CBC_SHA",
		[TLS_DH_RSA_WITH_AES_128_CBC_SHA] = "TLS_DH_RSA_WITH_AES_128_CBC_SHA",
		[TLS_DHE_DSS_WITH_AES_128_CBC_SHA] = "TLS_DHE_DSS_WITH_AES_128_CBC_SHA",
		[TLS_DHE_RSA_WITH_AES_128_CBC_SHA] = "TLS_DHE_RSA_WITH_AES_128_CBC_SHA",
		[TLS_DH_ANON_WITH_AES_128_CBC_SHA] = "TLS_DH_ANON_WITH_AES_128_CBC_SHA",
		[TLS_RSA_WITH_AES_256_CBC_SHA] = "TLS_RSA_WITH_AES_256_CBC_SHA",
		[TLS_DH_DSS_WITH_AES_256_CBC_SHA] = "TLS_DH_DSS_WITH_AES_256_CBC_SHA",
		[TLS_DH_RSA_WITH_AES_256_CBC_SHA] = "TLS_DH_RSA_WITH_AES_256_CBC_SHA",
		[TLS_DHE_DSS_WITH_AES_256_CBC_SHA] = "TLS_DHE_DSS_WITH_AES_256_CBC_SHA",
		[TLS_DHE_RSA_WITH_AES_256_CBC_SHA] = "TLS_DHE_RSA_WITH_AES_256_CBC_SHA",
		[TLS_DH_ANON_WITH_AES_256_CBC_SHA] = "TLS_DH_ANON_WITH_AES_256_CBC_SHA",
		[TLS_RSA_WITH_NULL_SHA256] = "TLS_RSA_WITH_NULL_SHA256",
		[TLS_RSA_WITH_AES_128_CBC_SHA256] = "TLS_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_RSA_WITH_AES_256_CBC_SHA256] = "TLS_RSA_WITH_AES_256_CBC_SHA256",
		[TLS_DH_DSS_WITH_AES_128_CBC_SHA256] = "TLS_DH_DSS_WITH_AES_128_CBC_SHA256",
		[TLS_DH_RSA_WITH_AES_128_CBC_SHA256] = "TLS_DH_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_DHE_DSS_WITH_AES_128_CBC_SHA256] = "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256",
		[TLS_RSA_WITH_CAMELLIA_128_CBC_SHA] = "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_RSA_EXPORT1024_WITH_RC4_56_MD5] = "TLS_RSA_EXPORT1024_WITH_RC4_56_MD5",
		[TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5] = "TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5",
		[TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA] = "TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA",
		[TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA] = "TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA",
		[TLS_RSA_EXPORT1024_WITH_RC4_56_SHA] = "TLS_RSA_EXPORT1024_WITH_RC4_56_SHA",
		[TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA] = "TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA",
		[TLS_DHE_DSS_WITH_RC4_128_SHA] = "TLS_DHE_DSS_WITH_RC4_128_SHA",
		[TLS_DHE_RSA_WITH_AES_128_CBC_SHA256] = "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_DH_DSS_WITH_AES_256_CBC_SHA256] = "TLS_DH_DSS_WITH_AES_256_CBC_SHA256",
		[TLS_DH_RSA_WITH_AES_256_CBC_SHA256] = "TLS_DH_RSA_WITH_AES_256_CBC_SHA256",
		[TLS_DHE_DSS_WITH_AES_256_CBC_SHA256] = "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256",
		[TLS_DHE_RSA_WITH_AES_256_CBC_SHA256] = "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256",
		[TLS_DH_ANON_WITH_AES_128_CBC_SHA256] = "TLS_DH_ANON_WITH_AES_128_CBC_SHA256",
		[TLS_DH_ANON_WITH_AES_256_CBC_SHA256] = "TLS_DH_ANON_WITH_AES_256_CBC_SHA256",
		[TLS_RSA_WITH_CAMELLIA_256_CBC_SHA] = "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_PSK_WITH_RC4_128_SHA] = "TLS_PSK_WITH_RC4_128_SHA",
		[TLS_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_PSK_WITH_AES_128_CBC_SHA] = "TLS_PSK_WITH_AES_128_CBC_SHA",
		[TLS_PSK_WITH_AES_256_CBC_SHA] = "TLS_PSK_WITH_AES_256_CBC_SHA",
		[TLS_DHE_PSK_WITH_RC4_128_SHA] = "TLS_DHE_PSK_WITH_RC4_128_SHA",
		[TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_DHE_PSK_WITH_AES_128_CBC_SHA] = "TLS_DHE_PSK_WITH_AES_128_CBC_SHA",
		[TLS_DHE_PSK_WITH_AES_256_CBC_SHA] = "TLS_DHE_PSK_WITH_AES_256_CBC_SHA",
		[TLS_RSA_PSK_WITH_RC4_128_SHA] = "TLS_RSA_PSK_WITH_RC4_128_SHA",
		[TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_RSA_PSK_WITH_AES_128_CBC_SHA] = "TLS_RSA_PSK_WITH_AES_128_CBC_SHA",
		[TLS_RSA_PSK_WITH_AES_256_CBC_SHA] = "TLS_RSA_PSK_WITH_AES_256_CBC_SHA",
		[TLS_RSA_WITH_SEED_CBC_SHA] = "TLS_RSA_WITH_SEED_CBC_SHA",
		[TLS_DH_DSS_WITH_SEED_CBC_SHA] = "TLS_DH_DSS_WITH_SEED_CBC_SHA",
		[TLS_DH_RSA_WITH_SEED_CBC_SHA] = "TLS_DH_RSA_WITH_SEED_CBC_SHA",
		[TLS_DHE_DSS_WITH_SEED_CBC_SHA] = "TLS_DHE_DSS_WITH_SEED_CBC_SHA",
		[TLS_DHE_RSA_WITH_SEED_CBC_SHA] = "TLS_DHE_RSA_WITH_SEED_CBC_SHA",
		[TLS_DH_ANON_WITH_SEED_CBC_SHA] = "TLS_DH_ANON_WITH_SEED_CBC_SHA",
		[TLS_RSA_WITH_AES_128_GCM_SHA256] = "TLS_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_RSA_WITH_AES_256_GCM_SHA384] = "TLS_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_DHE_RSA_WITH_AES_128_GCM_SHA256] = "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_DHE_RSA_WITH_AES_256_GCM_SHA384] = "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_DH_RSA_WITH_AES_128_GCM_SHA256] = "TLS_DH_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_DH_RSA_WITH_AES_256_GCM_SHA384] = "TLS_DH_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_DHE_DSS_WITH_AES_128_GCM_SHA256] = "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256",
		[TLS_DHE_DSS_WITH_AES_256_GCM_SHA384] = "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384",
		[TLS_DH_DSS_WITH_AES_128_GCM_SHA256] = "TLS_DH_DSS_WITH_AES_128_GCM_SHA256",
		[TLS_DH_DSS_WITH_AES_256_GCM_SHA384] = "TLS_DH_DSS_WITH_AES_256_GCM_SHA384",
		[TLS_DH_ANON_WITH_AES_128_GCM_SHA256] = "TLS_DH_ANON_WITH_AES_128_GCM_SHA256",
		[TLS_DH_ANON_WITH_AES_256_GCM_SHA384] = "TLS_DH_ANON_WITH_AES_256_GCM_SHA384",
		[TLS_PSK_WITH_AES_128_GCM_SHA256] = "TLS_PSK_WITH_AES_128_GCM_SHA256",
		[TLS_PSK_WITH_AES_256_GCM_SHA384] = "TLS_PSK_WITH_AES_256_GCM_SHA384",
		[TLS_DHE_PSK_WITH_AES_128_GCM_SHA256] = "TLS_DHE_PSK_WITH_AES_128_GCM_SHA256",
		[TLS_DHE_PSK_WITH_AES_256_GCM_SHA384] = "TLS_DHE_PSK_WITH_AES_256_GCM_SHA384",
		[TLS_RSA_PSK_WITH_AES_128_GCM_SHA256] = "TLS_RSA_PSK_WITH_AES_128_GCM_SHA256",
		[TLS_RSA_PSK_WITH_AES_256_GCM_SHA384] = "TLS_RSA_PSK_WITH_AES_256_GCM_SHA384",
		[TLS_PSK_WITH_AES_128_CBC_SHA256] = "TLS_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_PSK_WITH_AES_256_CBC_SHA384] = "TLS_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_PSK_WITH_NULL_SHA256] = "TLS_PSK_WITH_NULL_SHA256",
		[TLS_PSK_WITH_NULL_SHA384] = "TLS_PSK_WITH_NULL_SHA384",
		[TLS_DHE_PSK_WITH_AES_128_CBC_SHA256] = "TLS_DHE_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_DHE_PSK_WITH_AES_256_CBC_SHA384] = "TLS_DHE_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_DHE_PSK_WITH_NULL_SHA256] = "TLS_DHE_PSK_WITH_NULL_SHA256",
		[TLS_DHE_PSK_WITH_NULL_SHA384] = "TLS_DHE_PSK_WITH_NULL_SHA384",
		[TLS_RSA_PSK_WITH_AES_128_CBC_SHA256] = "TLS_RSA_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_RSA_PSK_WITH_AES_256_CBC_SHA384] = "TLS_RSA_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_RSA_PSK_WITH_NULL_SHA256] = "TLS_RSA_PSK_WITH_NULL_SHA256",
		[TLS_RSA_PSK_WITH_NULL_SHA384] = "TLS_RSA_PSK_WITH_NULL_SHA384",
		[TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_ECDH_ECDSA_WITH_NULL_SHA] = "TLS_ECDH_ECDSA_WITH_NULL_SHA",
		[TLS_ECDH_ECDSA_WITH_RC4_128_SHA] = "TLS_ECDH_ECDSA_WITH_RC4_128_SHA",
		[TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA] = "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA] = "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_NULL_SHA] = "TLS_ECDHE_ECDSA_WITH_NULL_SHA",
		[TLS_ECDHE_ECDSA_WITH_RC4_128_SHA] = "TLS_ECDHE_ECDSA_WITH_RC4_128_SHA",
		[TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA] = "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA] = "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDH_RSA_WITH_NULL_SHA] = "TLS_ECDH_RSA_WITH_NULL_SHA",
		[TLS_ECDH_RSA_WITH_RC4_128_SHA] = "TLS_ECDH_RSA_WITH_RC4_128_SHA",
		[TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDH_RSA_WITH_AES_128_CBC_SHA] = "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDH_RSA_WITH_AES_256_CBC_SHA] = "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_RSA_WITH_NULL_SHA] = "TLS_ECDHE_RSA_WITH_NULL_SHA",
		[TLS_ECDHE_RSA_WITH_RC4_128_SHA] = "TLS_ECDHE_RSA_WITH_RC4_128_SHA",
		[TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA] = "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA] = "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDH_ANON_WITH_NULL_SHA] = "TLS_ECDH_ANON_WITH_NULL_SHA",
		[TLS_ECDH_ANON_WITH_RC4_128_SHA] = "TLS_ECDH_ANON_WITH_RC4_128_SHA",
		[TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDH_ANON_WITH_AES_128_CBC_SHA] = "TLS_ECDH_ANON_WITH_AES_128_CBC_SHA",
		[TLS_ECDH_ANON_WITH_AES_256_CBC_SHA] = "TLS_ECDH_ANON_WITH_AES_256_CBC_SHA",
		[TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA] = "TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA",
		[TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA] = "TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA",
		[TLS_SRP_SHA_WITH_AES_128_CBC_SHA] = "TLS_SRP_SHA_WITH_AES_128_CBC_SHA",
		[TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA] = "TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA",
		[TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA] = "TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA",
		[TLS_SRP_SHA_WITH_AES_256_CBC_SHA] = "TLS_SRP_SHA_WITH_AES_256_CBC_SHA",
		[TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA] = "TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA",
		[TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA] = "TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDHE_PSK_WITH_RC4_128_SHA] = "TLS_ECDHE_PSK_WITH_RC4_128_SHA",
		[TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA] = "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA",
		[TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA] = "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256] = "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384] = "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_ECDHE_PSK_WITH_NULL_SHA] = "TLS_ECDHE_PSK_WITH_NULL_SHA",
		[TLS_ECDHE_PSK_WITH_NULL_SHA256] = "TLS_ECDHE_PSK_WITH_NULL_SHA256",
		[TLS_ECDHE_PSK_WITH_NULL_SHA384] = "TLS_ECDHE_PSK_WITH_NULL_SHA384",
		[SSL_RSA_FIPS_WITH_DES_CBC_SHA] = "SSL_RSA_FIPS_WITH_DES_CBC_SHA",
		[SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA] = "SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA",
		[SSL_RSA_FIPS_WITH_DES_CBC_SHA_2] = "SSL_RSA_FIPS_WITH_DES_CBC_SHA_2",
		[SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA_2] = "SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA_2",
	} &default="UNKNOWN";

	const x509_errors: table[count] of string = {
		[0]  = "ok",
		[1]  = "unable to get issuer cert",
		[2]  = "unable to get crl",
		[3]  = "unable to decrypt cert signature",
		[4]  = "unable to decrypt crl signature",
		[5]  = "unable to decode issuer public key",
		[6]  = "cert signature failure",
		[7]  = "crl signature failure",
		[8]  = "cert not yet valid",
		[9]  = "cert has expired",
		[10] = "crl not yet valid",
		[11] = "crl has expired",
		[12] = "error in cert not before field",
		[13] = "error in cert not after field",
		[14] = "error in crl last update field",
		[15] = "error in crl next update field",
		[16] = "out of mem",
		[17] = "depth zero self signed cert",
		[18] = "self signed cert in chain",
		[19] = "unable to get issuer cert locally",
		[20] = "unable to verify leaf signature",
		[21] = "cert chain too long",
		[22] = "cert revoked",
		[23] = "invalid ca",
		[24] = "path length exceeded",
		[25] = "invalid purpose",
		[26] = "cert untrusted",
		[27] = "cert rejected",
		[28] = "subject issuer mismatch",
		[29] = "akid skid mismatch",
		[30] = "akid issuer serial mismatch",
		[31] = "keyusage no certsign",
		[32] = "unable to get crl issuer",
		[33] = "unhandled critical extension"
		
	};

}
